#pragma once

#include <collector_ng/imap/operations/imap_operation.h>
#include <collector_ng/imap/operations/imap_init_operations.h>
#include <collector_ng/imap/operations/imap_list_operations.h>

#include <ymod_imapclient/call.h>

namespace yrpopper { namespace collector { namespace operations {

template <typename T>
using ImapCollectOpBuilder = ImapOpBuilder<T, CollectorHelperPtr>;

template <typename Res>
using ImapCollectOperation = GeneralImapOperation<CollectorHelperPtr, Res>;

class ImapInit : public ProxyOperation<ImapInitAsyncOperation>
{
public:
    ImapInit(AsyncCallback cb, CollectorHelperPtr helper)
        : ProxyOperation<ImapInitAsyncOperation>(cb), collectHelper(helper)

    {
    }

    void start() override
    {
        TASK_LOG(collectHelper->getContext(), info) << "ImapInit async op called";
        collectHelper->refreshImapClient();

        auto initHelper = std::make_shared<ImapInitHelper>(*collectHelper);
        initProxy(initHelper);
        run();
    }

protected:
    virtual std::exception_ptr onFinish(std::exception_ptr e) override
    {
        collectHelper->setReady(!e);
        return e;
    }

private:
    CollectorHelperPtr collectHelper;
};

class ImapConditionalInit : public ConditionalOperation<ImapInit>
{
public:
    ImapConditionalInit(AsyncCallback cb, CollectorHelperPtr helper)
        : ConditionalOperation<ImapInit>(cb, helper), helper(helper)
    {
    }

protected:
    virtual bool checkCondition() override
    {
        return !helper->isReady();
    }

private:
    CollectorHelperPtr helper;
};

class ImapNoop : public GeneralImapOperation<CollectorHelperPtr, ymod_imap_client::ImapResultPtr>
{
public:
    ImapNoop(AsyncCallback cb, CollectorHelperPtr helper)
        : GeneralImapOperation<CollectorHelperPtr, ymod_imap_client::ImapResultPtr>(cb, helper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        auto imapClient = helper->getImapClient();
        return imapClient->noop();
    }
};

class ImapConditionalNoop : public ConditionalOperation<ImapNoop>
{
public:
    ImapConditionalNoop(AsyncCallback cb, CollectorHelperPtr helper)
        : ConditionalOperation<ImapNoop>(cb, helper), helper(helper)
    {
    }

protected:
    virtual bool checkCondition() override
    {
        return helper->isReady();
    }

    virtual std::exception_ptr onFinish(std::exception_ptr e) override
    {
        if (e)
        {
            helper->setReady(false);
        }
        return std::exception_ptr();
    }

private:
    CollectorHelperPtr helper;
};

typedef ComplexOperation<
    CollectorHelperPtr,
    ImapCollectOpBuilder,

    ImapConditionalNoop,
    ImapConditionalInit>
    ImapInitConnection;

class ImapCollectFolders : public ProxyOperation<ImapCollectFoldersAsyncOperation>
{
public:
    ImapCollectFolders(AsyncCallback cb, CollectorHelperPtr helper)
        : ProxyOperation<ImapCollectFoldersAsyncOperation>(cb), helper(helper)
    {
    }

    void start() override
    {
        auto listHelper = std::make_shared<ImapListHelper>(*helper);

        initProxy(listHelper);
        run();
    }

protected:
    virtual std::exception_ptr onFinish(std::exception_ptr e) override
    {
        helper->setReady(!e);
        return e;
    }

private:
    CollectorHelperPtr helper;
};

typedef ComplexOperation<
    CollectorHelperPtr,
    ImapCollectOpBuilder,

    ImapInitConnection,
    ImapCollectFolders>
    ImapCollectBase;

class ImapCollect : public ImapCollectBase
{
public:
    ImapCollect(AsyncCallback cb, CollectorHelperPtr helper) : ImapCollectBase(cb, helper)
    {
    }

protected:
    virtual std::exception_ptr onFinish(std::exception_ptr e) override
    {
        if (e)
        {
            try
            {
                std::rethrow_exception(e);
            }
            catch (const ymod_imap_client::transport_error& err)
            {
                return std::make_exception_ptr(TransportError(err.what()));
            }
            catch (const ymod_imap_client::TimeoutException& err)
            {
                return std::make_exception_ptr(TimeoutError(err.what()));
            }
            catch (...)
            {
            }
        }
        return e;
    }
};

} // namespace operations
} // namespace collector
} // namespace yrpopper
