#pragma once

#include <collector_ng/imap/operations/imap_operation.h>
#include <collector_ng/imap/helpers/imap_folder_helper.h>

namespace yrpopper { namespace collector { namespace operations {

template <typename T>
using ImapFolderOpBuilder = ImapOpBuilder<T, ImapFolderHelperPtr>;

template <typename Res>
using ImapFolderOperation = GeneralImapOperation<ImapFolderHelperPtr, Res>;

typedef ImapFolderOperation<VoidResult> ImapFolderOperationNoRes;

class ImapFetchMessageChunk : public ImapFolderOperation<ymod_imap_client::MessageSetPtr>
{
public:
    ImapFetchMessageChunk(AsyncCallback cb, ImapFolderHelperPtr helper)
        : ImapFolderOperation<ymod_imap_client::MessageSetPtr>(cb, helper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        auto imapClient = helper->getImapClient();

        auto nextUid = helper->getImapFolder()->getUidnext();
        std::string from = boost::lexical_cast<std::string>(nextUid);

        lastUid = std::min(
            nextUid + helper->getSettings().fetchChunkSize,
            helper->getMailboxStats()->mailboxInfo.nextuid_ - 1);
        std::string to = boost::lexical_cast<std::string>(lastUid);

        TASK_LOG(helper->getContext(), info) << "Fetching new message chunk " << from << ":" << to;
        return imapClient->uidFetch(from + ":" + to, { ymod_imap_client::FetchArg::Uid });
    }

    virtual void process(ymod_imap_client::MessageSetPtr&& res) override
    {
        std::vector<int64_t> appendMessages;
        for (auto& msg : res->messages)
        {
            appendMessages.push_back(msg.uid);
        }

        helper->getImapFolder()->setUidnext(lastUid + 1);
        helper->setChunk(std::move(appendMessages));
    }

private:
    uint32_t lastUid = 0;
};

class ImapAddChunkToFolder : public ImapFolderOperationNoRes
{
public:
    ImapAddChunkToFolder(AsyncCallback cb, ImapFolderHelperPtr helper)
        : ImapFolderOperationNoRes(cb, helper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        auto imapFolder = helper->getImapFolder();
        return imapFolder->addInitialMessages(helper->getChunk());
    }
};

class ImapSaveFolder : public ImapFolderOperationNoRes
{
public:
    ImapSaveFolder(AsyncCallback cb, ImapFolderHelperPtr helper)
        : ImapFolderOperationNoRes(cb, helper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        imapFolder = helper->getImapFolder();
        if (helper->getImapFolder()->getUidnext() > helper->getMailboxStats()->mailboxInfo.nextuid_)
        {
            helper->getImapFolder()->setUidnext(helper->getMailboxStats()->mailboxInfo.nextuid_);
        }
        return imapFolder->save();
    }

    virtual std::exception_ptr onFinish(std::exception_ptr e) override
    {
        if (e)
        {
            imapFolder->invalidate();
        }
        return e;
    }

private:
    ImapFolderPtr imapFolder;
};

typedef ComplexOperation<
    ImapFolderHelperPtr,
    ImapFolderOpBuilder,

    ImapFetchMessageChunk,
    ImapAddChunkToFolder,
    ImapSaveFolder>
    ImapProcessFolderChunk;

class ImapFetchFolderMessagesAsyncOperation : public CycleOperation<ImapProcessFolderChunk>
{
public:
    ImapFetchFolderMessagesAsyncOperation(AsyncCallback cb, ImapFolderHelperPtr helper)
        : CycleOperation<ImapProcessFolderChunk>(cb, helper), helper(helper)
    {
    }

protected:
    virtual bool checkCondition() override
    {
        return helper->getImapFolder()->getUidnext() <
            helper->getMailboxStats()->mailboxInfo.nextuid_;
    }

private:
    ImapFolderHelperPtr helper;
};

} // namespace operations
} // namespace collector
} // namespace yrpopper
