#pragma once

#include <memory>

#include <collector_ng/imap/operations/imap_operation.h>
#include <collector_ng/imap/helpers/imap_init_helper.h>
#include <collector_ng/collector_error.h>

#include <boost/algorithm/string.hpp>

#include <oauth/oauth.h>

namespace yrpopper { namespace collector { namespace operations {

template <typename T>
using ImapInitOpBuilder = ImapOpBuilder<T, ImapInitHelperPtr>;

template <typename Res>
using ImapInitOperation = GeneralImapOperation<ImapInitHelperPtr, Res>;

typedef ImapInitOperation<ymod_imap_client::ImapResultPtr> ImapInitOperationNoRes;

class ImapConnect : public ImapInitOperation<ymod_imap_client::ConnectResultPtr>
{
public:
    ImapConnect(AsyncCallback cb, ImapInitHelperPtr helper)
        : ImapInitOperation<ymod_imap_client::ConnectResultPtr>(cb, helper)
    {
    }

    virtual ~ImapConnect()
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        auto imapClient = helper->getImapClient();
        auto context = helper->getContext();

        return imapClient->connect(
            context->task->server, context->task->port, context->task->use_ssl);
    }

    virtual void process(ymod_imap_client::ConnectResultPtr&& res) override
    {
        helper->getContext()->host_ip = res->serverIp;
    }

    virtual std::exception_ptr onFinish(std::exception_ptr e) override
    {
        if (e)
        {
            try
            {
                std::rethrow_exception(e);
            }
            catch (const ymod_imap_client::resolve_error& e)
            {
                return std::make_exception_ptr(ResolveError(e.what()));
            }
            catch (const std::exception& e)
            {
                return std::make_exception_ptr(ConnectError(e.what()));
            }
        }
        return e;
    }
};

class ImapCapability : public ImapInitOperation<ymod_imap_client::ImapCapabilityPtr>
{
public:
    ImapCapability(AsyncCallback cb, ImapInitHelperPtr initHelper)
        : ImapInitOperation<ymod_imap_client::ImapCapabilityPtr>(cb, initHelper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        auto imapClient = helper->getImapClient();
        return imapClient->capability();
    }

    virtual void process(ymod_imap_client::ImapCapabilityPtr&& res) override
    {
        auto serverInfo = helper->getServerInfo();
        serverInfo->setStarttls(res->capability.starttls);
        serverInfo->setId(res->capability.id);
        serverInfo->setAuthPlain(res->capability.authPlain);
        serverInfo->setGmail(res->capability.gmailEx);
    }

    virtual std::exception_ptr imapOnFinish(std::exception_ptr /* e */) override
    {
        return std::exception_ptr();
    }
};

class ImapStartTls : public ImapInitOperationNoRes
{
public:
    ImapStartTls(AsyncCallback cb, ImapInitHelperPtr initHelper)
        : ImapInitOperationNoRes(cb, initHelper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        auto imapClient = helper->getImapClient();
        return imapClient->startTls();
    }

    virtual std::exception_ptr onFinish(std::exception_ptr e) override
    {
        if (e)
        {
            try
            {
                std::rethrow_exception(e);
            }
            catch (const std::exception& e)
            {
                return std::make_exception_ptr(SslError(e.what()));
            }
        }
        return e;
    }
};

class ImapConditionalStartTls : public ConditionalOperation<ImapStartTls>
{
public:
    ImapConditionalStartTls(AsyncCallback cb, ImapInitHelperPtr initHelper)
        : ConditionalOperation<ImapStartTls>(cb, initHelper), helper(initHelper)
    {
    }

protected:
    virtual bool checkCondition() override
    {
        return helper->getServerInfo()->canStarttls() && !helper->getContext()->task->use_ssl;
    }

private:
    ImapInitHelperPtr helper;
};

class ImapId : public ImapInitOperation<ymod_imap_client::IdResultPtr>
{
public:
    ImapId(AsyncCallback cb, ImapInitHelperPtr initHelper)
        : ImapInitOperation<ymod_imap_client::IdResultPtr>(cb, initHelper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        auto imapClient = helper->getImapClient();
        return imapClient->id(); // TODO config
    }

    virtual void process(ymod_imap_client::IdResultPtr&& res) override
    {
        auto serverInfo = helper->getServerInfo();
        auto& settings = helper->getSettings();
        auto serverName = res->data["name"];

        ymod_imap_client::ImapClient::OauthLoginType loginType =
            ymod_imap_client::ImapClient::Multiline;
        for (auto& loginInfo : settings.oauthLoginTypes)
        {
            if (boost::algorithm::contains(serverName, loginInfo.serverName))
            {
                loginType = loginInfo.literal ? ymod_imap_client::ImapClient::SinglelineLiteral :
                                                ymod_imap_client::ImapClient::SinglelineQuoted;
                break;
            }
        }
        serverInfo->setOauthLoginType(loginType);
    }

    virtual std::exception_ptr imapOnFinish(std::exception_ptr /* e */) override
    {
        return std::exception_ptr();
    }
};

class ImapConditionalId : public ConditionalOperation<ImapId>
{
public:
    ImapConditionalId(AsyncCallback cb, ImapInitHelperPtr initHelper)
        : ConditionalOperation<ImapId>(cb, initHelper), helper(initHelper)
    {
    }

protected:
    virtual bool checkCondition() override
    {
        return helper->getServerInfo()->canId();
    }

private:
    ImapInitHelperPtr helper;
};

class ImapLogin : public ImapInitOperationNoRes
{
public:
    ImapLogin(AsyncCallback cb, ImapInitHelperPtr initHelper)
        : ImapInitOperationNoRes(cb, initHelper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        auto imapClient = helper->getImapClient();
        auto context = helper->getContext();

        return imapClient->login(context->task->login, context->task->password);
    }
};

class ImapAuthPlain : public ImapInitOperationNoRes
{
public:
    ImapAuthPlain(AsyncCallback cb, ImapInitHelperPtr initHelper)
        : ImapInitOperationNoRes(cb, initHelper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        auto imapClient = helper->getImapClient();
        auto context = helper->getContext();

        return imapClient->authPlain(context->task->login, context->task->password);
    }
};

class ImapToken : public ImapInitOperation<std::string>
{
public:
    ImapToken(AsyncCallback cb, ImapInitHelperPtr initHelper)
        : ImapInitOperation<std::string>(cb, initHelper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        auto oauthModule = yplatform::find<oauth::OauthService>("oauth_module");
        auto context = helper->getContext();

        return oauthModule->getAccessToken(
            context, context->task->server, context->task->oauth_refresh_token);
    }

    virtual void process(std::string&& res) override
    {
        helper->setToken(std::move(res));
    }
};

class ImapXoauth2 : public ImapInitOperationNoRes
{
public:
    ImapXoauth2(AsyncCallback cb, ImapInitHelperPtr initHelper)
        : ImapInitOperationNoRes(cb, initHelper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        auto imapClient = helper->getImapClient();
        auto context = helper->getContext();

        auto loginType = helper->getServerInfo()->getOauthLoginType();
        return imapClient->loginOauth(context->task->login, helper->getToken(), loginType);
    }
};

typedef ComplexOperation<ImapInitHelperPtr, ImapInitOpBuilder, ImapToken, ImapXoauth2> ImapOauth;

class ImapAuth : public AsyncOperation
{
public:
    ImapAuth(AsyncCallback cb, ImapInitHelperPtr initHelper)
        : AsyncOperation(cb), helper(initHelper)
    {
    }

protected:
    virtual void run() override
    {
        auto context = helper->getContext();
        auto self = sharedAs<ImapAuth>();
        auto callback = [self](std::exception_ptr e) { self->finishOperation(e); };
        if (context->task->oauth_refresh_token.empty())
        {
            if (helper->getServerInfo()->canAuthPlain())
            {
                runAsync<ImapAuthPlain>(callback, helper);
            }
            else
            {
                runAsync<ImapLogin>(callback, helper);
            }
        }
        else
        {
            runAsync<ImapOauth>(callback, helper);
        }
    }

    virtual std::exception_ptr onFinish(std::exception_ptr e) override
    {
        if (e)
        {
            try
            {
                std::rethrow_exception(e);
            }
            catch (const std::exception& e)
            {
                return std::make_exception_ptr(AuthError(e.what()));
            }
        }
        return e;
    }

private:
    ImapInitHelperPtr helper;
};

class ImapLoadFoldersDb : public ImapInitOperation<VoidResult>
{
public:
    ImapLoadFoldersDb(AsyncCallback cb, ImapInitHelperPtr initHelper)
        : ImapInitOperation<VoidResult>(cb, initHelper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        auto imapClient = helper->getImapClient();
        auto context = helper->getContext();

        return helper->getImapFolderList()->init();
    }

    virtual std::exception_ptr onFinish(std::exception_ptr e) override
    {
        if (e)
        {
            try
            {
                std::rethrow_exception(e);
            }
            catch (const std::exception& e)
            {
                return std::make_exception_ptr(DbError(e.what()));
            }
        }
        return e;
    }
};

typedef ComplexOperation<
    ImapInitHelperPtr,
    ImapInitOpBuilder,

    ImapConnect,
    ImapCapability,
    ImapConditionalId,
    ImapConditionalStartTls,
    ImapAuth,
    ImapCapability,
    ImapLoadFoldersDb>
    ImapInitAsyncOperation;

} // namespace operations
} // namespace collector
} // namespace yrpopper
