#pragma once

#include <collector_ng/imap/operations/imap_operation.h>
#include <collector_ng/imap/operations/imap_collect_folders_operations.h>
#include <collector_ng/imap/operations/imap_prepared_folder_operations.h>

#include <collector_ng/imap/helpers/imap_list_helper.h>

namespace yrpopper { namespace collector { namespace operations {

template <typename T>
using ImapListOpBuilder = ImapOpBuilder<T, ImapListHelperPtr>;

template <typename Res>
using ImapListOperation = GeneralImapOperation<ImapListHelperPtr, Res>;

class ImapList : public ImapListOperation<ymod_imap_client::ImapListPtr>
{
public:
    ImapList(AsyncCallback cb, ImapListHelperPtr helper)
        : ImapListOperation<ymod_imap_client::ImapListPtr>(cb, helper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        auto imapClient = helper->getImapClient();
        return imapClient->list("", "*");
    }

    virtual void process(ymod_imap_client::ImapListPtr&& res) override
    {
        helper->setServerMailboxList(std::move(res->mailboxes));
    }
};

class ImapNormalFilteredList : public ImapList
{
public:
    ImapNormalFilteredList(AsyncCallback cb, ImapListHelperPtr helper) : ImapList(cb, helper)
    {
    }

protected:
    virtual void process(ymod_imap_client::ImapListPtr&& list) override
    {
        using namespace ymod_imap_client;
        ImapMailboxList res;

        for (auto& folder : list->mailboxes)
        {
            if ((folder->flags & ListResponse::ff_noselect) != ListResponse::ff_noselect)
            {
                res.push_back(folder);
            }
        }

        helper->setServerMailboxList(std::move(res));
    }
};

class ImapGmailFilteredList : public ImapList
{
public:
    ImapGmailFilteredList(AsyncCallback cb, ImapListHelperPtr helper) : ImapList(cb, helper)
    {
    }

protected:
    virtual void process(ymod_imap_client::ImapListPtr&& list) override
    {
        using namespace ymod_imap_client;
        ImapMailboxList res;

        bool folderAllExists = false;
        for (auto& folder : list->mailboxes)
        {
            if ((folder->flags & ListResponse::ff_all) == ListResponse::ff_all)
            {
                folderAllExists = true;
                break;
            }
        }

        for (auto& folder : list->mailboxes)
        {
            if ((folder->flags & ListResponse::ff_noselect) == ListResponse::ff_noselect)
            {
                continue;
            }

            if (folderAllExists)
            {
                if (folder->flags &
                    (ListResponse::ff_all | ListResponse::ff_trash | ListResponse::ff_spam))
                {
                    res.push_back(folder);
                }
            }
            else
            {
                if (!(folder->flags &
                      (ListResponse::ff_all | ListResponse::ff_important |
                       ListResponse::ff_starred)))
                {
                    res.push_back(folder);
                }
            }
        }

        helper->setServerMailboxList(std::move(res));
    }
};

class ImapFilteredList : public AsyncOperation
{
public:
    ImapFilteredList(AsyncCallback cb, ImapListHelperPtr helper)
        : AsyncOperation(cb), helper(helper)
    {
    }

protected:
    virtual void run() override
    {
        TASK_LOG(helper->getContext(), info) << "ImapFilteredList async op called";
        auto self = sharedAs<ImapFilteredList>();
        auto callback = [self](std::exception_ptr e) { self->finishOperation(e); };
        if (helper->getServerInfo()->isGmail())
        {
            runAsync<ImapGmailFilteredList>(callback, helper);
        }
        else
        {
            runAsync<ImapNormalFilteredList>(callback, helper);
        }
    }

private:
    ImapListHelperPtr helper;
};

class ImapFoldersCycleProxy : public ProxyOperation<ImapFoldersCycle>
{
public:
    ImapFoldersCycleProxy(AsyncCallback cb, ImapListHelperPtr helper)
        : ProxyOperation<ImapFoldersCycle>(cb), listHelper(helper)
    {
    }

    void start()
    {
        auto collectFoldersHelper = std::make_shared<ImapCollectFoldersHelper>(
            *listHelper, listHelper->getServerMailboxList());

        initProxy(collectFoldersHelper);
        run();
    }

private:
    ImapListHelperPtr listHelper;
};

class ImapPreparedFoldersCycleProxy : public ProxyOperation<ImapPreparedFoldersCycle>
{
public:
    ImapPreparedFoldersCycleProxy(AsyncCallback cb, ImapListHelperPtr helper)
        : ProxyOperation<ImapPreparedFoldersCycle>(cb), listHelper(helper)
    {
    }

    void start() override
    {
        auto preparedFoldersHelper = std::make_shared<ImapPreaparedFolderHelper>(*listHelper);

        initProxy(preparedFoldersHelper);
        run();
    }

protected:
    virtual std::exception_ptr onFinish(std::exception_ptr e) override
    {
        if (!e)
        {
            listHelper->getImapFolderList()->markProcessed();
        }
        return e;
    }

private:
    ImapListHelperPtr listHelper;
};

typedef ComplexOperation<
    ImapListHelperPtr,
    ImapListOpBuilder,

    ImapFilteredList,
    ImapFoldersCycleProxy,
    ImapPreparedFoldersCycleProxy>
    ImapCollectFoldersAsyncOperation;

} // namespace operations
} // namespace collector
} // namespace yrpopper
