#pragma once

#include <collector_ng/collector_error.h>
#include <collector_ng/imap/operations/imap_operation.h>
#include <collector_ng/imap/helpers/imap_message_helper.h>

#include <common/typed_log.h>

#include <boost/algorithm/string.hpp>

#include <ymod_imapclient/errors.h>

namespace yrpopper { namespace collector { namespace operations {

template <typename T>
using ImapMessageOpBuilder = ImapOpBuilder<T, ImapMessageHelperPtr>;

template <typename Res>
using ImapMessageOperation = GeneralImapOperation<ImapMessageHelperPtr, Res>;

class ImapLoadMessages : public ImapMessageOperation<imap_uidl_map_ptr>
{
public:
    ImapLoadMessages(AsyncCallback cb, ImapMessageHelperPtr helper)
        : ImapMessageOperation<imap_uidl_map_ptr>(cb, helper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        auto imapFolder = helper->getMessageFolder();
        auto limit = imapFolder->getMessageLimit();
        if (limit < 1)
        {
            limit = 1;
        }

        return imapFolder->loadMessages(helper->getMessageState(), limit);
    }

    virtual void process(imap_uidl_map_ptr&& messages) override
    {
        auto imapFolder = helper->getMessageFolder();
        imapFolder->updateLimit(messages->size());
        helper->setMessages(std::deque<uint32_t>(messages->rbegin(), messages->rend()));
    }
};

class ImapFetchMessage : public ImapMessageOperation<ymod_imap_client::MessageSetPtr>
{
public:
    ImapFetchMessage(AsyncCallback cb, ImapMessageHelperPtr helper)
        : ImapMessageOperation<ymod_imap_client::MessageSetPtr>(cb, helper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        auto imapClient = helper->getImapClient();
        auto uid = boost::lexical_cast<string>(helper->getCurrentMessage());
        helper->setProcessorMessage(createMessage(uid));

        ymod_imap_client::FetchArgs args = { ymod_imap_client::FetchArg::Flags,
                                             ymod_imap_client::FetchArg::Body };
        if (helper->getServerInfo()->isGmail())
        {
            args.insert(ymod_imap_client::FetchArg::GmailLabels);
        }
        return imapClient->uidFetch(uid, args);
    }

    processor::MessagePtr createMessage(const std::string& imapId)
    {
        auto message = std::make_shared<processor::Message>();
        message->imapId = imapId;
        auto folder = helper->getMessageFolder();
        message->srcFolder = folder->getName().asString();
        return message;
    }

    virtual void process(ymod_imap_client::MessageSetPtr&& res) override
    {
        if (res->messages.empty())
        {
            throw NoSuchMessageError("No such message");
        }

        auto message = helper->getProcessorMessage();
        auto folder = helper->getMessageFolder();
        message->dstDelim = folder->getName().delim;

        auto msg = res->messages.front();
        message->answered = msg.flags & ymod_imap_client::FetchResponse::mf_answered;
        message->deleted = msg.flags & ymod_imap_client::FetchResponse::mf_deleted;
        message->flagged = msg.flags & ymod_imap_client::FetchResponse::mf_flagged;
        message->seen = msg.flags & ymod_imap_client::FetchResponse::mf_seen;

        if (helper->getServerInfo()->isGmail())
        {
            processGmailLabels(message, msg.gmailLabels);
        }
        else if (helper->getContext()->task->root_folder.empty())
        {
            auto flags = folder->getFlags();
            if (flags & ymod_imap_client::ListResponse::ff_drafts)
            {
                message->dstFolder = "\\Drafts";
                message->spamFolder = "\\Drafts";
            }
            else if (flags & ymod_imap_client::ListResponse::ff_sent)
            {
                message->dstFolder = "\\Sent";
                message->spamFolder = "\\Sent";
            }
            else if (flags & ymod_imap_client::ListResponse::ff_trash)
            {
                message->dstFolder = "\\Trash";
                message->spamFolder = "\\Trash";
            }
            else if (
                flags &
                (ymod_imap_client::ListResponse::ff_spam |
                 ymod_imap_client::ListResponse::ff_inbox))
            {
                message->dstFolder = "\\Inbox";
                message->spamFolder = "\\Spam";
            }

            if (message->dstFolder.empty())
            {
                message->dstFolder = mapFolderName(folder->getName());
            }
        }
        else
        {
            auto rootFolder = helper->getContext()->task->root_folder;
            std::replace(rootFolder.begin(), rootFolder.end(), '|', message->dstDelim);
            message->dstFolder =
                rootFolder + std::string(&message->dstDelim, 1) + folder->getName().asString();
            message->spamFolder = message->dstFolder;
        }

        message->body = std::make_shared<string>(std::move(msg.body));
        message->start = message->body->begin();
        message->end = message->body->end();
    }

    std::string mapFolderName(const ymod_imap_client::Utf8MailboxName& name)
    {
        auto strName = name.asString();
        auto pos = strName.find(name.delim);
        auto topFolder = strName.substr(0, pos);

        auto& settings = helper->getSettings();
        auto it = settings.folderNamesMap.find(topFolder);

        if (it != settings.folderNamesMap.end())
        {
            auto res = it->second;
            if (pos != string::npos)
            {
                res += strName.substr(pos, string::npos);
            }
            return res;
        }
        return strName;
    }

    void processGmailLabels(
        processor::MessagePtr message,
        const std::vector<ymod_imap_client::Utf8Label>& gmailLables)
    {
        auto rootFolder = helper->getContext()->task->root_folder;
        std::replace(rootFolder.begin(), rootFolder.end(), '|', '/');

        for (auto& label : gmailLables)
        {
            auto strLabel = label.asString();
            if (strLabel[0] != '\\')
            {
                if (boost::contains(strLabel, "/"))
                {
                    std::vector<std::string> parts;
                    boost::split(parts, strLabel, boost::is_any_of(string(1, '/')));
                    if (!parts.empty())
                    {
                        auto it = parts.begin();
                        std::string sublabel = *it++;
                        message->labels.push_back(sublabel);
                        while (it != parts.end())
                        {
                            sublabel += ("/" + *it++);
                            message->labels.push_back(sublabel);
                        }
                    }
                }
                else
                {
                    message->labels.push_back(strLabel);
                }
            }
            else
            {
                if (strLabel == "\\Starred")
                {
                    message->flagged = true;
                    continue;
                }

                if (strLabel == "\\Important")
                {
                    continue;
                }

                if (strLabel == "\\Draft")
                {
                    strLabel = "\\Drafts";
                }

                if (rootFolder.empty())
                {
                    message->dstFolder = strLabel;
                }
                else
                {
                    std::string folder = strLabel.substr(1, string::npos);
                    message->dstFolder = rootFolder + string("/") + folder;
                    message->spamFolder = message->dstFolder;
                }
            }
        }
    }
};

class ImapSendMessage : public ImapMessageOperation<processor::ProcessorResult>
{
public:
    ImapSendMessage(AsyncCallback cb, ImapMessageHelperPtr helper)
        : ImapMessageOperation<processor::ProcessorResult>(cb, helper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        helper->getContext()->sent_count++;

        auto message = helper->getProcessorMessage();
        auto& label_id = helper->getContext()->task->label_id;
        if (label_id.size())
        {
            message->lids.push_back(label_id);
        }

        auto processor = yplatform::find<processor::ProcessorService>("message_processor");
        return processor->processMessage(helper->getContext(), message);
    }

    virtual void process(processor::ProcessorResult&& res) override
    {
        helper->setMessageProcessingResult(res);
    }
};

typedef ComplexOperation<
    ImapMessageHelperPtr,
    ImapMessageOpBuilder,

    ImapFetchMessage,
    ImapSendMessage>
    ImapProcessMessageBase;

class ImapProcessMessage : public ImapProcessMessageBase
{
public:
    ImapProcessMessage(AsyncCallback cb, ImapMessageHelperPtr helper)
        : ImapProcessMessageBase(cb, helper), helper(helper)
    {
    }

protected:
    virtual std::exception_ptr onFinish(std::exception_ptr e) override
    {
        reportProcessMessage(e);
        helper->setMessageProcessingError(e);
        return std::exception_ptr();
    }

    void reportProcessMessage(std::exception_ptr e)
    {
        auto status = e ? "error"s : "success"s;
        auto message = helper->getProcessorMessage();
        auto srcMessageId = message ? "imap_id:" + message->imapId : "";
        auto srcFolderName = message ? message->srcFolder : "";
        auto dstFolderName = message ? message->dstFolder : "";
        auto& processingResult = helper->getMessageProcessingResult();
        std::string reason;
        if (e)
        {
            std::string imapResponse;
            unpackError(e, reason, imapResponse);
            if (imapResponse.size()) reason += ", server response: "s + imapResponse;
        }
        typed_log::log_store_message(
            helper->getContext(),
            status,
            reason,
            dstFolderName,
            srcFolderName,
            srcMessageId,
            "imap",
            processingResult.messageId,
            processingResult.smtpResponse);
    }

    void unpackError(std::exception_ptr e, std::string& reason, std::string& imapResponse)
    {
        try
        {
            std::rethrow_exception(e);
        }
        catch (const ymod_imap_client::ImapException& error)
        {
            reason = error.what();
            imapResponse = error.serverResponse ? *error.serverResponse : ""s;
        }
        catch (const std::exception& error)
        {
            reason = error.what();
            imapResponse = ""s;
        }
    }

private:
    ImapMessageHelperPtr helper;
};

class ImapMarkMessage : public ImapMessageOperation<VoidResult>
{
public:
    ImapMarkMessage(AsyncCallback cb, ImapMessageHelperPtr helper)
        : ImapMessageOperation<VoidResult>(cb, helper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        auto folder = helper->getMessageFolder();
        ImapFolder::MessageState newState = ImapFolder::collectedState;

        auto error = helper->getMessageProcessingError();
        if (error)
        {
            newState =
                (helper->getMessageState() == ImapFolder::initialState ?
                     ImapFolder::errorState :
                     ImapFolder::notCollectedState);
        }
        return folder->editMessage(helper->getCurrentMessage(), newState);
    }

    virtual std::exception_ptr onFinish(std::exception_ptr e) override
    {
        auto error = helper->getMessageProcessingError();
        if (error)
        {
            return error;
        }
        return e;
    }
};

typedef ComplexOperation<
    ImapMessageHelperPtr,
    ImapMessageOpBuilder,

    ImapProcessMessage,
    ImapMarkMessage>
    ImapCollectMessage;

class ImapProcessMessageIteration : public ProxyOperation<ImapCollectMessage>
{
public:
    ImapProcessMessageIteration(AsyncCallback cb, ImapMessageHelperPtr helper)
        : ProxyOperation<ImapCollectMessage>(cb), helper(helper)
    {
    }

    void start() override
    {
        TASK_LOG(helper->getContext(), info)
            << "Processing message " << helper->getCurrentMessage();
        initProxy(helper);
        run();
    }

protected:
    virtual std::exception_ptr onFinish(std::exception_ptr err) override
    {
        if (err)
        {
            if (helper->getContext()->is_cancelled())
            {
                return err;
            }

            try
            {
                std::rethrow_exception(err);
            }
            catch (const ymod_imap_client::NoException& e)
            {
                TASK_LOG(helper->getContext(), error)
                    << "ImapProcessMessageIteration exception: " << e.what();
                if (helper->getMessageState() == ImapFolder::initialState)
                {
                    err = std::exception_ptr();
                }
            }
            catch (const ymod_imap_client::BadException& e)
            {
                TASK_LOG(helper->getContext(), error)
                    << "ImapProcessMessageIteration exception: " << e.what();
                if (helper->getMessageState() == ImapFolder::initialState)
                {
                    err = std::exception_ptr();
                }
            }
            catch (const processor::ProcessorError& e)
            {
                TASK_LOG(helper->getContext(), error)
                    << "ImapProcessMessageIteration exception: " << e.what();
                if (helper->getMessageState() == ImapFolder::initialState)
                {
                    err = std::exception_ptr();
                }
            }
            catch (const NoSuchMessageError& e)
            {
                err = std::exception_ptr();
            }
            catch (const std::exception& e)
            {
                return err;
            }
        }

        helper->nextMessage();
        return err;
    }

private:
    ImapMessageHelperPtr helper;
};

class ImapProcessMessageCycled : public CycleOperation<ImapProcessMessageIteration>
{
public:
    ImapProcessMessageCycled(AsyncCallback cb, ImapMessageHelperPtr helper)
        : CycleOperation<ImapProcessMessageIteration>(cb, helper), helper(helper)
    {
    }

protected:
    virtual bool checkCondition() override
    {
        return helper->hasMoreMessages();
    }

private:
    ImapMessageHelperPtr helper;
};

typedef ComplexOperation<
    ImapMessageHelperPtr,
    ImapMessageOpBuilder,

    ImapLoadMessages,
    ImapProcessMessageCycled>
    ImapProcessMessages;

} // namespace operations
} // namespace collector
} // namespace yrpopper
