#pragma once

#include <collector_ng/imap/operations/imap_operation.h>
#include <collector_ng/imap/operations/imap_message_operations.h>

#include <collector_ng/imap/helpers/imap_prepared_folder_helper.h>

namespace yrpopper { namespace collector { namespace operations {

template <typename T>
using ImapPreparedFolderOpBuilder = ImapOpBuilder<T, ImapPreaparedFolderHelperPtr>;

template <typename Res>
using ImapPreparedFolderOperation = GeneralImapOperation<ImapPreaparedFolderHelperPtr, Res>;

class ImapSelectFolder : public ImapPreparedFolderOperation<ymod_imap_client::ImapMailboxResultPtr>
{
public:
    ImapSelectFolder(AsyncCallback cb, ImapPreaparedFolderHelperPtr helper)
        : ImapPreparedFolderOperation<ymod_imap_client::ImapMailboxResultPtr>(cb, helper)
    {
    }

protected:
    virtual FutureType getFuture() override
    {
        TASK_LOG(helper->getContext(), info)
            << "Retrying errors for folder \"" << helper->getCurrentFolder()->getName().asString()
            << "\"";

        auto imapClient = helper->getImapClient();
        return imapClient->examine(helper->getCurrentFolder()->getName());
    }
};

class ImapProcessPreparedFolder : public ProxyOperation<ImapProcessMessages>
{
public:
    ImapProcessPreparedFolder(AsyncCallback cb, ImapPreaparedFolderHelperPtr helper)
        : ProxyOperation<ImapProcessMessages>(cb), helper(helper)
    {
    }

    void start() override
    {
        auto messageHelper = std::make_shared<ImapMessageHelper>(
            *helper, helper->getCurrentFolder(), ImapFolder::errorState);

        initProxy(messageHelper);
        run();
    }

protected:
    virtual std::exception_ptr onFinish(std::exception_ptr err) override
    {
        if (err)
        {
            try
            {
                std::rethrow_exception(err);
            }
            catch (const ymod_imap_client::NoException& e)
            {
                TASK_LOG(helper->getContext(), error)
                    << "ImapProcessPreparedFolder exception: " << e.what();
                err = std::exception_ptr();
            }
            catch (const ymod_imap_client::BadException& e)
            {
                TASK_LOG(helper->getContext(), error)
                    << "ImapProcessPreparedFolder exception: " << e.what();
                err = std::exception_ptr();
            }
            catch (const processor::ProcessorError& e)
            {
                TASK_LOG(helper->getContext(), error)
                    << "ImapProcessPreparedFolder exception: " << e.what();
                err = std::exception_ptr();
            }
            catch (const std::exception& e)
            {
                return err;
            }
        }

        return err;
    }

private:
    ImapPreaparedFolderHelperPtr helper;
};

typedef ComplexOperation<
    ImapPreaparedFolderHelperPtr,
    ImapPreparedFolderOpBuilder,

    ImapSelectFolder,
    ImapProcessPreparedFolder>
    ImapProcessPrepared;

class ImapPreparedFoldersIteration : public ProxyOperation<ImapProcessPrepared>
{
public:
    ImapPreparedFoldersIteration(AsyncCallback cb, ImapPreaparedFolderHelperPtr helper)
        : ProxyOperation<ImapProcessPrepared>(cb), helper(helper)
    {
    }

    void start() override
    {
        initProxy(helper);
        run();
    }

protected:
    virtual std::exception_ptr onFinish(std::exception_ptr e) override
    {
        if (!e)
        {
            helper->nextFolder();
        }
        return e;
    }

private:
    ImapPreaparedFolderHelperPtr helper;
};

class ImapPreparedFoldersCycle : public CycleOperation<ImapPreparedFoldersIteration>
{
public:
    ImapPreparedFoldersCycle(AsyncCallback cb, ImapPreaparedFolderHelperPtr helper)
        : CycleOperation<ImapPreparedFoldersIteration>(cb, helper), helper(helper)
    {
    }

protected:
    virtual bool checkCondition() override
    {
        return helper->hasMoreFolders();
    }

private:
    ImapPreaparedFolderHelperPtr helper;
};

} // namespace operations
} // namespace collector
} // namespace yrpopper
