#pragma once

#include <exception>
#include <functional>
#include <memory>

namespace yrpopper {

typedef std::function<void(std::exception_ptr)> AsyncCallback;

class AsyncOperation : public std::enable_shared_from_this<AsyncOperation>
{
protected:
public:
    AsyncOperation(AsyncCallback cb) : finishCallback(cb)
    {
    }

    virtual ~AsyncOperation()
    {
    }

    virtual void start()
    {
        run();
    }

protected:
    virtual void run() = 0;
    virtual void cleanup()
    {
    }
    virtual std::exception_ptr onFinish(std::exception_ptr e)
    {
        return e;
    }

    template <class SiblingClass>
    std::shared_ptr<SiblingClass> sharedAs()
    {
        return std::dynamic_pointer_cast<SiblingClass>(shared_from_this());
    }

    void finishOperation(std::exception_ptr e = std::exception_ptr())
    {
        e = onFinish(e);
        cleanup();

        if (finishCallback)
        {
            finishCallback(e);
            finishCallback = AsyncCallback();
        }
    }

private:
    AsyncCallback finishCallback;
};
typedef std::shared_ptr<AsyncOperation> AsyncOperationPtr;

typedef std::function<void()> OperationRunner;

template <typename Operation, typename... Args>
void runAsync(Args&&... args)
{
    auto op = std::make_shared<Operation>(std::forward<Args>(args)...);
    op->start();
}

template <typename Operation, typename... Args>
OperationRunner makeRunner(Args&&... args)
{
    return std::bind(&runAsync<Operation, Args...>, std::forward<Args>(args)...);
}

} // namespace yrpopper
