#pragma once

#include <common/task.h>
#include <common/uidl.h>
#include <common/data_source.h>
#include <yplatform/log.h>
#include <yplatform/log/typed.h>

#include <yplatform/task_context.h>
#include <list>
#include <atomic>

namespace yrpopper {

using PlatformContextPtr = yplatform::task_context_ptr;
using PromiseVoidResultPtr = std::shared_ptr<promise_void_t>;

struct rpop_context
    : public yplatform::task_context
    , private boost::noncopyable
{
    explicit rpop_context(
        task_ptr t,
        const string& uniq_id,
        bool verbose_run,
        bool clean_run,
        PromiseVoidResultPtr manual_run_promise_ptr)
        : yplatform::task_context()
        , task(t)
        , sent_count(0)
        , sent_size(0)
        , success_finished(false)
        , verbose(verbose_run)
        , clean(clean_run)
        , manual_run_promise_ptr(manual_run_promise_ptr)
    {
        if (uniq_id.size())
        {
            append_to_uniq_id(uniq_id);
        }
        last_updated = boost::posix_time::microsec_clock::local_time();
    }

    virtual const std::string& get_name() const
    {
        static const std::string NAME = "rpop_session";
        return NAME;
    }

    void update_ctx()
    {
        last_updated = boost::posix_time::microsec_clock::local_time();
    }

    task_ptr task;
    std::size_t sent_count;
    std::size_t sent_size;
    bool success_finished;
    string host_ip;
    boost::posix_time::ptime last_updated;
    std::atomic_bool verbose;
    std::atomic_bool clean;

    PromiseVoidResultPtr manual_run_promise_ptr;

    mutable yplatform::log::source errorLogger;

protected:
    virtual ptree_ptr core_get_stat() const
    {
        ptree_ptr result = yplatform::task_context::core_get_stat();
        result->put("account.popid", task->popid);
        result->put("account.language", task->bb_info.getLanguage());
        result->put("account.suid", task->suid);
        result->put("account.storage", task->dbname);
        result->put("account.email", task->bb_info.getEmail());
        result->put("server.host", task->server);
        result->put("server.port", task->port);
        result->put("server.login", task->login);
        result->put("server.ssl", task->use_ssl);
        result->put("server.leave_msgs", task->leave_msgs);
        result->put("server.ip", host_ip);
        result->put("server.imap", task->use_imap);
        result->put("task.last_conn", task->last_connect);
        result->put("task.session_duration", task->session_duration);
        result->put("task.create_date", task->create_date);
        result->put("task.bad_retries", task->bad_retries);
        result->put("last_updated", boost::posix_time::to_simple_string(last_updated));
        return result;
    }
};

typedef boost::shared_ptr<rpop_context> rpop_context_ptr;

struct sched_context
    : public yplatform::task_context
    , private boost::noncopyable
{
public:
    virtual const std::string& get_name() const
    {
        static const std::string NAME = "scheduler";
        return NAME;
    }
};

typedef boost::shared_ptr<sched_context> sched_context_ptr;

// Log helpers
#define YRIMAP_ERROR(contextPtr) LERR_(contextPtr->errorLogger, contextPtr)
#define YRIMAP_LOG(contextPtr) LINFO_(contextPtr)

#define YRIMAP_DISPATCH_LOG(context) LINFO_(YGLOBAL_LOGGER, context) << "[dispatch] "

#define YRIMAP_DISPATCH_ERROR(context) LERR_(YGLOBAL_LOGGER, context) << "[dispatch] "

} // namespace yrpopper
