#pragma once

#include <memory>
#include <sstream>
#include <common/context.h>

#include <yplatform/find.h>
#include <ymod_httpclient/call.h>

#include <boost/property_tree/json_parser.hpp>
#include <boost/property_tree/xml_parser.hpp>

using yplatform::task_context_ptr;

namespace yrpopper {

typedef ymod_http_client::string_ptr http_string_ptr;

struct HttpGet
{
    static ymod_http_client::future_void_t request(
        yplatform::task_context_ptr ctx,
        ymod_http_client::response_handler_ptr handler,
        const ymod_http_client::remote_point_info_ptr host,
        const std::string& request,
        const http_string_ptr& /* postData */, // GET should ignore POST data
        const std::string& headers = "")
    {
        auto httpService = yplatform::find<yhttp::call>("http_client");
        return httpService->get_url(ctx, handler, host, request, headers);
    }
};

struct HttpPost
{
    static ymod_http_client::future_void_t request(
        yplatform::task_context_ptr ctx,
        ymod_http_client::response_handler_ptr handler,
        const ymod_http_client::remote_point_info_ptr host,
        const std::string& request,
        const http_string_ptr& postData,
        const std::string& headers = "")
    {
        auto httpService = yplatform::find<yhttp::call>("http_client");
        return httpService->post_url(ctx, handler, host, request, postData, headers);
    }
};

template <typename Method>
class HttpLoader
    : public ymod_http_client::response_handler
    , public boost::enable_shared_from_this<HttpLoader<Method>>
{
public:
    HttpLoader(task_context_ptr context, const string& host, size_t maxRetries)
        : context(context), maxRetries(maxRetries)
    {
        try
        {
            auto httpService = yplatform::find<yhttp::call>("http_client");
            remotePointInfo = httpService->make_rm_info(host);
        }
        catch (const std::exception& e)
        {
            TASK_LOG(context, error) << "HttpLoader construct error: " << e.what();
        }
        catch (...)
        {
        }
    }

    virtual ~HttpLoader()
    {
    }

    virtual ymod_http_client::handler_version version() const
    {
        return ymod_http_client::handler_version_already_parse_body;
    }

    virtual void handle_data(const char* chunk, unsigned long long size)
    {
        data << std::move(string(chunk, size));
    }

    virtual void set_code(int code, const string& description)
    {
        httpCode = code;
        httpCodeDescription = description;
    }

    void load()
    {
        retry = 0;
        makeRequest();
    }

    template <class SiblingClass>
    boost::shared_ptr<SiblingClass> sharedAs()
    {
        return boost::dynamic_pointer_cast<SiblingClass>(this->shared_from_this());
    }

protected:
    virtual string getRequestData()
    {
        return "";
    }
    virtual http_string_ptr getPostData()
    {
        return http_string_ptr();
    }
    virtual string getHeaders()
    {
        return "";
    }

    virtual void handleData() = 0;
    virtual void handleError(const std::string& message) = 0;

private:
    void makeRequest()
    {
        if (!remotePointInfo)
        {
            handleError("Malformed HttpLoader");
            return;
        }
        resetData();

        httpFuture.reset();
        auto requestData = getRequestData();
        auto postData = getPostData();
        auto headers = getHeaders();

        auto future = Method::request(
            context, this->shared_from_this(), remotePointInfo, requestData, postData, headers);
        httpFuture.reset(new ymod_http_client::future_void_t(std::move(future)));

        httpFuture->add_callback(
            boost::bind(&HttpLoader<Method>::onResponse, this->shared_from_this()));
    }

    void onResponse()
    {
        try
        {
            httpFuture->get();
            if (httpCode != 200)
            {
                auto message =
                    boost::lexical_cast<std::string>(httpCode) + " " + httpCodeDescription;
                if (httpCode / 100 == 5)
                {
                    onError(message);
                }
                else
                {
                    // Retrying of non 5XX has no sense.
                    handleError(message);
                }
                return;
            }
            handleData();
        }
        catch (std::exception& e)
        {
            return onError(e.what());
        }
        catch (...)
        {
            return onError("Unknown exception");
        }
    }

    void onError(const string& errorMessage)
    {
        TASK_LOG(context, error) << "HttpLoader error: " << errorMessage;

        bool shouldRetry = ++retry < maxRetries;
        if (shouldRetry)
        {
            makeRequest();
        }
        else
        {
            handleError(errorMessage);
        }
    }

    void resetData()
    {
        data.str("");
    }

protected:
    task_context_ptr context;
    ymod_http_client::remote_point_info_ptr remotePointInfo;
    const size_t maxRetries;
    size_t retry = 0;
    std::shared_ptr<ymod_http_client::future_void_t> httpFuture;
    std::stringstream data;
    int httpCode = 0;
    std::string httpCodeDescription;
};

template <typename Method>
class JsonHttpLoader : public HttpLoader<Method>
{
public:
    JsonHttpLoader(task_context_ptr context, const string& host, size_t maxRetries)
        : HttpLoader<Method>(context, host, maxRetries)
    {
    }

    virtual ~JsonHttpLoader()
    {
    }

protected:
    virtual void handleJson(boost::property_tree::ptree&& jsonPtree) = 0;

    virtual void handleData()
    {
        try
        {
            boost::property_tree::ptree jsonPtree;
            boost::property_tree::read_json(this->data.seekg(std::ios_base::beg), jsonPtree);
            handleJson(std::move(jsonPtree));
        }
        catch (const std::exception& e)
        {
            this->handleError(
                std::string("Got exception in JsonHttpLoader::handleData: ") + e.what());
        }
    }
};

template <typename Method>
class XmlHttpLoader : public HttpLoader<Method>
{
public:
    XmlHttpLoader(task_context_ptr context, const string& host, size_t maxRetries)
        : HttpLoader<Method>(context, host, maxRetries)
    {
    }

    virtual ~XmlHttpLoader()
    {
    }

protected:
    virtual void handleXml(boost::property_tree::ptree&& xmlPtree) = 0;

    virtual void handleData()
    {
        try
        {
            boost::property_tree::ptree xmlPtree;
            boost::property_tree::read_xml(this->data.seekg(std::ios_base::beg), xmlPtree);
            handleXml(std::move(xmlPtree));
        }
        catch (const std::exception& e)
        {
            this->handleError(
                std::string("Got exception in XmlHttpLoader::handleData: ") + e.what());
        }
    }
};

} // namespace yrpopper
