#pragma once

#include <lib/idna.h>
#include <common/config.h>
#include <boost/noncopyable.hpp>

namespace yrpopper {

namespace detail {

template <typename T>
class c_raii_ptr : private boost::noncopyable
{
public:
    c_raii_ptr() : ptr_(0)
    {
    }
    ~c_raii_ptr()
    {
        if (ptr_) free(ptr_);
    }

    T* get() const
    {
        return ptr_;
    }

    T*& get()
    {
        return ptr_;
    }

private:
    T* ptr_;
};

}

inline string idna_decode(const string& text)
{
    detail::c_raii_ptr<char> output;
    idna_to_unicode_8z8z(text.c_str(), &output.get(), 0);
    return string(output.get());
}

inline string idna_encode(const string& text)
{
    detail::c_raii_ptr<char> output;
    int rc = idna_to_ascii_8z(text.c_str(), &output.get(), 0);
    if (rc != IDNA_SUCCESS)
        throw std::runtime_error(string("invalid idna string: ") + idna_strerror((Idna_rc)rc));
    return string(output.get());
}

inline string idna_encode_email(const string& email)
{
    std::size_t s = email.find('@');
    if (s == string::npos) return idna_encode(email);
    else
        return idna_encode(email.substr(0, s)) + "@" + idna_encode(email.substr(s + 1));
}

inline string idna_decode_email(const string& email)
{
    std::size_t s = email.find('@');
    if (s == string::npos) return idna_decode(email);
    else
        return idna_decode(email.substr(0, s)) + "@" + idna_decode(email.substr(s + 1));
}

}
