#pragma once

#include <vector>
#include <algorithm>
#include <boost/bind.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/asio/ip/address.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/algorithm/string/split.hpp>

namespace yrpopper { namespace common {

namespace detail {

struct ip_net
{
    boost::asio::ip::address min;
    boost::asio::ip::address max;
};

struct ip_in_net
{
    explicit ip_in_net(const boost::asio::ip::address& _ip) : ip_(_ip){};

    bool operator()(const detail::ip_net& _net) const
    {
        return (_net.max > ip_) && (_net.min < ip_);
    };

    boost::asio::ip::address ip_;
};

}

class ip_nets
{

public:
    typedef std::pair<boost::asio::ip::address, unsigned> net_description;

    static net_description parse(const std::string& net)
    {
        try
        {
            std::vector<std::string> components;
            boost::split(components, net, boost::algorithm::is_from_range('/', '/'));
            return std::make_pair(
                boost::asio::ip::address::from_string(components.at(0)),
                boost::lexical_cast<unsigned>(components.at(1)));
        }
        catch (const std::bad_alloc& e)
        {
            throw;
        }

        throw std::runtime_error("invalid ip net");
    }

    template <typename NetSet>
    void set_net(const NetSet& nets)
    {
        nets_.clear();
        nets_.reserve(nets.size());

        for (typename NetSet::const_iterator i = nets.begin(), i_end = nets.end(); i != i_end; ++i)
        {
            detail::ip_net net = { apply_mask(i->first, i->second, false),
                                   apply_mask(i->first, i->second, true) };
            nets_.push_back(net);
        }
    }

    bool contains(const std::string& addr)
    {
        return contains(boost::asio::ip::address::from_string(addr));
    }

    bool contains(const boost::asio::ip::address& addr)
    {
        std::vector<detail::ip_net>::const_iterator it =
            std::find_if(nets_.begin(), nets_.end(), detail::ip_in_net(addr));

        return it != nets_.end();
    }

private:
    // false - min, true - max
    boost::asio::ip::address apply_mask(
        const boost::asio::ip::address& addr,
        unsigned mask,
        bool op)
    {
        if (addr.is_v4()) return apply_mask(addr.to_v4(), mask, op);
        else if (addr.is_v6())
            return apply_mask(addr.to_v6(), mask, op);
        throw std::runtime_error("unsupported ip type");
    }

    template <typename T1, typename T2>
    T1 min(T1 t1, T2 t2)
    {
        return t1 < t2 ? t1 : t2;
    }

    template <typename T>
    boost::asio::ip::address apply_mask(const T& addr, unsigned _mask, bool op)
    {
        typename T::bytes_type bytes = addr.to_bytes();
        uint8_t mask = static_cast<uint8_t>(_mask);
        assert(8 * bytes.size() > mask);
        mask = static_cast<uint8_t>(8 * bytes.size() - mask);
        typename T::bytes_type::reverse_iterator i = bytes.rbegin();
        while (mask)
        {
            uint8_t mask_byte = 0xFF;
            if (op)
            {
                mask_byte = static_cast<uint8_t>(
                    mask_byte >> min(static_cast<uint8_t>(8u) - mask, static_cast<uint8_t>(8u)));
                *i |= mask_byte;
            }
            else
            {
                mask_byte = static_cast<uint8_t>(mask_byte << min(mask, static_cast<uint8_t>(8u)));
                *i &= mask_byte;
            }
            mask = static_cast<uint8_t>(mask - min(mask, static_cast<uint8_t>(8u)));
            ++i;
        }
        return T(bytes);
    }

    std::vector<detail::ip_net> nets_;
};

}}
