#pragma once

#include <common/config.h>
#include <yplatform/exception.h>
#include <boost/lexical_cast.hpp>
#include <contrib/libs/yajl/api/yajl_gen.h>

namespace yrpopper { namespace api { namespace http {

struct json_builder_error
    : YPLATFORM_ERROR_DEF(
          json_builder_error,
          yplatform::exception,
          "json_builder_error",
          "json build error");

class json_builder
{
public:
    explicit json_builder()
    {

        if ((writer_ = yajl_gen_alloc(NULL)) == NULL) throw_error("cannot create JSON generator");

        yajl_gen_config(writer_, yajl_gen_beautify, 0);
    }

    ~json_builder()
    {
        yajl_gen_free(writer_);
    }

    json_builder& begin_map()
    {
        check_status(yajl_gen_map_open(writer_));
        return *this;
    }

    json_builder& begin_map(const std::string& _name)
    {
        check_status(yajl_gen_string(
            writer_, reinterpret_cast<const unsigned char*>(_name.c_str()), _name.length()));
        check_status(yajl_gen_map_open(writer_));
        return *this;
    }

    json_builder& end_map()
    {
        check_status(yajl_gen_map_close(writer_));
        return *this;
    }

    json_builder& value(const std::string& _name, long unsigned _value)
    {
        check_status(yajl_gen_string(
            writer_, reinterpret_cast<const unsigned char*>(_name.c_str()), _name.length()));
        check_status(yajl_gen_integer(writer_, _value));
        return *this;
    }

    json_builder& value(const std::string& _name, const std::string& _value)
    {
        check_status(yajl_gen_string(
            writer_, reinterpret_cast<const unsigned char*>(_name.c_str()), _name.length()));
        check_status(yajl_gen_string(
            writer_, reinterpret_cast<const unsigned char*>(_value.c_str()), _value.length()));
        return *this;
    }

    json_builder& value(const std::string& _name, bool _value)
    {
        check_status(yajl_gen_string(
            writer_, reinterpret_cast<const unsigned char*>(_name.c_str()), _name.length()));
        check_status(yajl_gen_bool(writer_, _value));
        return *this;
    }

    json_builder& begin_array()
    {
        check_status(yajl_gen_array_open(writer_));
        return *this;
    }

    json_builder& begin_array(const std::string& _name)
    {
        check_status(yajl_gen_string(
            writer_, reinterpret_cast<const unsigned char*>(_name.c_str()), _name.length()));
        check_status(yajl_gen_array_open(writer_));
        return *this;
    }

    json_builder& end_array()
    {
        check_status(yajl_gen_array_close(writer_));
        return *this;
    }

    json_builder& array_value(const std::string& _value)
    {
        check_status(yajl_gen_string(
            writer_, reinterpret_cast<const unsigned char*>(_value.c_str()), _value.length()));
        return *this;
    }

    json_builder& array_value(bool _value)
    {
        check_status(yajl_gen_bool(writer_, _value));
        return *this;
    }

    void check_status(yajl_gen_status status)
    {
        if (status != yajl_gen_status_ok) throw_error("JSON generator error");
    }

    json_builder& close()
    {
        check_status(yajl_gen_get_buf(writer_, &buffer_, &length_));
        return *this;
    }

    std::size_t result_size() const
    {
        return length_;
    }

    const char* result()
    {
        if (buffer_ == NULL)
        {
            throw_error("close json builder first");
        }

        return reinterpret_cast<const char*>(buffer_);
    }

    std::string buf_;

private:
    void throw_error(const string& reason)
    {
        throw json_builder_error("desc", reason, "");
    }
    yajl_gen writer_;

    const unsigned char* buffer_;
    size_t length_;
};

}}}
