#pragma once

#include "task_error.h"

#include <common/util.h>
#include <boost/shared_ptr.hpp>
#include <boost/date_time.hpp>
#include <boost/thread/recursive_mutex.hpp>
#include <boost/thread/locks.hpp>
#include <boost/unordered_map.hpp>
#include <list>
#include <queue>
#include <yplatform/log.h>

#include <mutex>

namespace yrpopper {

struct base_task
{
    base_task() : session_duration(0), bad_retries(0), validated(1), abook_sync_state(0)
    {
    }
    base_task(time_t dr, std::size_t br)
        : session_duration(dr), bad_retries(br), validated(1), abook_sync_state(0)
    {
    }

    time_t session_duration;
    std::size_t bad_retries;
    unsigned validated;
    unsigned abook_sync_state;
    std::string uidl_hash;
};

struct base_task_status
{
    base_task_status() : error(code::ok), start_time(0), message_count(0), mailbox_sync(false)
    {
    }
    base_task_status(error ec, time_t st, std::size_t mc)
        : error(ec), start_time(st), message_count(mc), mailbox_sync(false)
    {
    }

    bool is_correct()
    {
        return !error;
    }

    error error;
    time_t start_time;
    std::size_t message_count;
    std::string server_response;
    bool mailbox_sync;
};

struct task_status
    : public base_task_status
    , public base_task
{
    task_status()
    {
    }
    task_status(yrpopper::error ec, time_t st, time_t dr, std::size_t mc, std::size_t br)
        : base_task_status(ec, st, mc), base_task(dr, br)
    {
    }
};

typedef boost::shared_ptr<task_status> task_status_ptr;

struct base_task_info : public base_task
{
    base_task_info()
        : popid(0)
        , port(0)
        , use_ssl(false)
        , last_connect(0)
        , touch_time(0)
        , leave_msgs(true)
        , mark_archive_read(false)
        , use_imap(false)
        , root_folder()
    {
    }

    popid_t popid;
    std::string server;
    unsigned port;
    std::string login;
    std::string password;
    std::string oauth_refresh_token;
    bool use_ssl;
    std::string email;
    std::time_t last_connect;
    std::time_t touch_time;
    bool leave_msgs;
    bool mark_archive_read;
    action_list_t action;
    bool use_imap;
    std::string root_folder;
    std::string label_id;
};

class blackbox_task_info
{
    typedef std::lock_guard<std::mutex> Locker;

public:
    blackbox_task_info() : lastUpdated(0)
    {
    }

    std::string getUid()
    {
        Locker lock(guard);
        return uid;
    }
    std::string getMdb()
    {
        Locker lock(guard);
        return mdb;
    }
    std::string getEmail()
    {
        Locker lock(guard);
        return email;
    }
    std::string getLanguage()
    {
        Locker lock(guard);
        return language;
    }
    time_t getLastUpdated()
    {
        Locker lock(guard);
        return lastUpdated;
    }

    void update(std::string uid, std::string mdb, std::string email, std::string language)
    {
        Locker lock(guard);

        this->uid = uid;
        this->mdb = mdb;
        this->email = email;
        this->language = language;

        this->lastUpdated = std::time(0);
    }

    void update(const blackbox_task_info& other)
    {
        Locker otherLock(other.guard);
        Locker thisLock(guard);

        this->uid = other.uid;
        this->mdb = other.mdb;
        this->email = other.email;
        this->language = other.language;
        this->lastUpdated = other.lastUpdated;
    }

private:
    std::string uid;
    std::string mdb;
    std::string email;
    std::string language;
    time_t lastUpdated;

    mutable std::mutex guard;
};

struct task_info : public base_task_info
{
    task_info() : is_on(0), last_msg_count(0)
    {
    }
    uint16_t is_on;
    std::string error_status;
    std::size_t last_msg_count;
};
typedef std::shared_ptr<task_info> task_info_ptr;

struct task : public base_task_info
{
    typedef boost::recursive_mutex mutex_t;
    typedef boost::unique_lock<mutex_t> lock_t;

    task()
        : create_date(boost::posix_time::max_date_time)
        , abook_sync_date(boost::posix_time::max_date_time)
        , chunk_id(0)
    {
    }

    std::string uid;
    std::string suid;
    std::time_t create_date;
    std::time_t abook_sync_date;
    std::string dbname;
    uint64_t chunk_id;

    blackbox_task_info bb_info;

    error error;
    std::size_t last_msg_count = 0;
    uint64_t is_on = 0;

    void refresh_data(const boost::shared_ptr<task_status>& status, uint64_t new_is_on)
    {
        lock_t lock(mux_);
        bad_retries = status->bad_retries;
        validated = status->validated;
        abook_sync_state = status->abook_sync_state;
        uidl_hash = status->uidl_hash;
        last_msg_count = status->message_count;
        error = status->error;
        is_on = new_is_on;
    }

    void refresh_data(const boost::shared_ptr<task>& task)
    {
        lock_t lock(mux_);
        server = task->server;
        port = task->port;
        login = task->login;
        password = task->password;
        email = task->email;
        bad_retries = task->bad_retries;
        session_duration = task->session_duration;
        create_date = task->create_date;
        mark_archive_read = task->mark_archive_read;
        use_ssl = task->use_ssl;
        use_imap = task->use_imap;
        root_folder = task->root_folder;
        label_id = task->label_id;
        leave_msgs = task->leave_msgs;
        validated = task->validated;
        abook_sync_state = task->abook_sync_state;
        uidl_hash = task->uidl_hash;
        oauth_refresh_token = task->oauth_refresh_token;
    }

    std::string to_log_string() const
    {
        lock_t lock(mux_);
        std::ostringstream stream;
        stream << "popid=" << popid << ", suid=" << suid << ", server=" << server << ":" << port
               << ", login=" << login << ", validated=" << validated << ", use_ssl=" << use_ssl
               << ", leave_msgs=" << leave_msgs << ", mark_archive_read=" << mark_archive_read
               << ", abook_sync_state=" << getAbookState()
               << ", oauth=" << !oauth_refresh_token.empty() << ", is_on=" << is_on;

        return stream.str();
    }

    std::string getAbookState() const
    {
        std::string state;
        switch (abook_sync_state)
        {
        case 1:
            state = "disable";
            break;
        case 2:
            state = "enable";
            break;
        case 3:
            state = "unsupported";
            break;
        case 4:
            state = "done";
            break;
        default:
            state = "error";
            break;
        }
        state += "(" + boost::lexical_cast<std::string>(abook_sync_state) + ")";
        return state;
    }

private:
    mutable mutex_t mux_;
};

typedef boost::shared_ptr<task> task_ptr;
typedef std::list<task_ptr> task_ptr_list;
typedef boost::shared_ptr<task_ptr_list> task_ptr_list_ptr;

}
