#pragma once

#include <sstream>
#include <ctime>
#include <set>
#include <boost/date_time.hpp>
#include <openssl/evp.h>
#include <vector>
#include <stdexcept>
#include <fstream>
#include <arpa/inet.h>
#include <common/config.h>
#include <ymod_imapclient/errors.h>
#include <yplatform/exception.h>
#include <yplatform/encoding/base64.h>

#include <blowfish.h>

namespace yrpopper {

inline string get_exception_reason(std::exception_ptr e)
{
    std::string reason = "no exceptions";
    if (!e) return reason;

    try
    {
        std::rethrow_exception(e);
    }
    catch (const ymod_imap_client::ImapException& error)
    {
        reason = error.what();
        string imapResponse = error.serverResponse ? *error.serverResponse : "";
        if (imapResponse.size()) reason += ", server response: "s + imapResponse;
    }
    catch (yplatform::exception& e)
    {
        reason = e.private_message();
    }
    catch (const std::exception& error)
    {
        reason = error.what();
    }
    catch (...)
    {
        reason = "unknown";
    }
    return reason;
}

template <typename Future>
string get_exception_reason(const Future f)
{
    return get_exception_reason(f.get_exception());
}

template <typename PTime>
int to_unix_time(const PTime& t)
{
    std::tm ctm = boost::posix_time::to_tm(t);
    return std::mktime(&ctm);
}

template <typename PTime>
void to_local_time(time_t ut, PTime& pt)
{
    pt = boost::posix_time::from_time_t(ut);
    pt += boost::posix_time::seconds(ut - to_unix_time(pt));
}

template <typename Hook>
void load_line_file(const string& file, const string& desc, Hook f)
{
    std::ifstream stream(file.c_str());
    if (!stream.is_open() || !stream.good()) throw std::runtime_error(desc + " file not found");
    stream >> std::noskipws;
    string line;
    int count = 0;
    while (!stream.eof())
    {
        std::getline(stream, line);
        if (line.empty()) continue;
        f(line);
        ++count;
    }
    if (count == 0) throw std::runtime_error(desc + " file is empty");
}

inline void insert_string_2_set(const string& v, std::set<string>& s)
{
    s.insert(v);
}

template <class Range>
inline std::string make_string(const Range& r)
{
    return std::string(r.begin(), r.end());
}

inline std::string pad(const std::string& in, size_t size, char ch)
{
    std::string padded(size, ch);
    std::copy(in.begin(), in.begin() + std::min(size, in.size()), padded.begin());
    return padded;
}

inline std::string bf_encrypt(const std::string& in, const std::string& key, const std::string& iv)
{
    return blowfish::encrypt(in, key, iv);
}

inline std::string bf_decrypt(const std::string& in, const std::string& key, const std::string& iv)
{
    return blowfish::decrypt(in, key, iv);
}

namespace legacy {

// True iff the password is unencrypted or legacy-encrypted
inline bool is_encrypted(const std::string& ins)
{
    enum
    {
        PASSWORD_IS_ENCRYPTED = 9
    };

    if (ins.size() < 2) return true;

    if (ins[0] != PASSWORD_IS_ENCRYPTED) return true;

    if (ins[1] != PASSWORD_IS_ENCRYPTED) return true;

    return false;
}

inline std::string encrypt_password(const std::string& in, const std::string& key)
{
    try
    {
        enum
        {
            PASSWORD_IS_ENCRYPTED = 9
        };

        const std::string iv = "password";

        std::string r(bf_encrypt(in, pad(key, 16, 's'), iv));
        return char(PASSWORD_IS_ENCRYPTED) +
            make_string(yplatform::base64_encode(r.begin(), r.end()));
    }
    catch (std::exception& e)
    {
        std::ostringstream err;
        err << "encryption error (" << e.what() << "), pass: " << in << ", key: " << key;
        throw std::runtime_error(err.str());
    }
}

inline std::string decrypt_password(const std::string& ins, const std::string& key)
{
    try
    {
        enum
        {
            PASSWORD_IS_ENCRYPTED = 9
        };

        const std::string iv = "password";

        // Is the password encrypted?
        if (ins[0] != PASSWORD_IS_ENCRYPTED) return ins;

        std::string pkey(pad(key, 16, 's'));
        std::string in = make_string(yplatform::base64_decode(ins.begin() + 1, ins.end()));
        return bf_decrypt(in, pkey, iv);
    }
    catch (std::exception& e)
    {
        std::ostringstream err;
        err << "decryption error (" << e.what() << "), pass: " << ins << ", key: " << key;
        throw std::runtime_error(err.str());
    }
}
}

typedef std::map<uint32_t, std::string> versioned_keys_t;

inline versioned_keys_t compute_derived_keys(const std::string& keys_filename)
{
    return blowfish::password::compute_derived_keys(keys_filename);
}

inline std::string decrypt_password(
    const std::string& in64,
    const versioned_keys_t& dkeys,
    const std::string& iv)
{
    if (legacy::is_encrypted(in64)) return legacy::decrypt_password(in64, iv);
    return blowfish::password::decrypt(in64, dkeys, iv);
}

inline std::string encrypt_password(
    const std::string& in,
    const versioned_keys_t& dkeys,
    const std::string& iv)
{
    return blowfish::password::encrypt(in, dkeys, iv);
}

} // namespace yrpopper {
