#pragma once

#include <common/config.h>
#include <yplatform/exception.h>
#include <boost/lexical_cast.hpp>
#include <libxml/xmlwriter.h>

namespace yrpopper { namespace api { namespace http {

struct xml_builder_error
    : YPLATFORM_ERROR_DEF(
          xml_builder_error,
          yplatform::exception,
          "xml_builder_error",
          "xml build error");

class xml_builder
{
public:
    explicit xml_builder(const string& encoding = "utf-8")
    {
        if ((buf_ = xmlBufferCreate()) == 0) throw_error("can`t create buffer");
        if ((writer_ = xmlNewTextWriterMemory(buf_, 0)) == 0)
        {
            xmlBufferFree(buf_);
            throw_error("can`t create writer");
        }
        int rc = xmlTextWriterStartDocument(writer_, NULL, encoding.c_str(), NULL);
        if (rc < 0)
        {
            xmlFreeTextWriter(writer_);
            xmlBufferFree(buf_);
            throw_error("can`t create document");
        }
    }
    ~xml_builder()
    {
        xmlFreeTextWriter(writer_);
        xmlBufferFree(buf_);
    }

    xml_builder& begin_node(const string& name)
    {
        int rc = xmlTextWriterStartElement(writer_, BAD_CAST name.c_str());
        if (rc < 0) throw_error("can`t begin node");
        return *this;
    }
    xml_builder& add_attr(const string& name, const string& value)
    {
        int rc =
            xmlTextWriterWriteAttribute(writer_, BAD_CAST name.c_str(), BAD_CAST value.c_str());
        if (rc < 0) throw_error("can`t add attribute");
        return *this;
    }
    template <typename T>
    xml_builder& add_attr(const string& name, T value)
    {
        return add_attr(name, boost::lexical_cast<string>(value));
    }
    xml_builder& print_body(const string& body)
    {
        int rc = xmlTextWriterWriteString(writer_, BAD_CAST body.c_str());
        if (rc < 0) throw_error("can`t print body");
        return *this;
    }
    template <typename T>
    xml_builder& print_body(T body)
    {
        return print_body(boost::lexical_cast<string>(body));
    }
    xml_builder& end_node()
    {
        int rc = xmlTextWriterEndElement(writer_);
        if (rc < 0) throw_error("can`t end node");
        return *this;
    }
    xml_builder& endl()
    {
        int rc = xmlTextWriterWriteRaw(writer_, BAD_CAST "\n");
        if (rc < 0) throw_error("can`t end node");
        return *this;
    }
    xml_builder& close()
    {
        int rc = xmlTextWriterEndDocument(writer_);
        if (rc < 0) throw_error("can`t close document");
        return *this;
    }
    std::size_t result_size() const
    {
        return buf_->use;
    }
    const char* result() const
    {
        return (const char*)buf_->content;
    }

private:
    void throw_error(const string& reason)
    {
        throw xml_builder_error("desc", reason, xmlGetLastError()->message);
    }
    xmlTextWriterPtr writer_;
    xmlBufferPtr buf_;
};

}}}
