#include <db/api_interface.h>
#include <db/query/request_query.h>
#include <db/query/execute_query.h>

#include <db/handlers/task_info_handler.h>
#include <db/handlers/new_popid.h>
#include <db/handlers/rpop_owner.h>
#include <db/handlers/http_status_handler.h>
#include <db/handlers/status_handler.h>
#include <db/handlers/imap_folders.h>

#include <yplatform/find.h>

static const boost::posix_time::ptime DATE_UNREAD_ARCHIVE =
    boost::posix_time::time_from_string("2000-01-01 00:00:00");

namespace yrpopper::db {

ApiInterface::ApiInterface()
{
    dispatcher = yplatform::find<query_dispatcher>("query_dispatcher");
}

future_task_info_list ApiInterface::list(PlatformContextPtr ctx, const std::string& suid)
{
    using ListRequestQuery = query::RequestQuery<task_info_list_ptr, handlers::task_info_handler>;
    auto req = std::make_shared<ListRequestQuery>(ctx, "api_list");

    req->addArgs(suid);
    return dispatcher->run_on_all(req, request_target::try_replica);
}

future_task_info_list ApiInterface::info(
    PlatformContextPtr ctx,
    const std::string& suid,
    popid_t popid)
{
    using InfoRequestQuery = query::RequestQuery<task_info_list_ptr, handlers::task_info_handler>;
    auto req = std::make_shared<InfoRequestQuery>(ctx, "api_info");

    req->addArgs(suid, popid);
    return dispatcher->run(req, popid, request_target::try_replica);
}

future_task_info_list ApiInterface::info(
    PlatformContextPtr ctx,
    popid_t popid)
{
    using InfoRequestQuery = query::RequestQuery<task_info_list_ptr, handlers::task_info_handler>;
    auto req = std::make_shared<InfoRequestQuery>(ctx, "api_collector_info");

    req->addArgs(popid);
    return dispatcher->run(req, popid, request_target::try_replica);
}

future_popid_t ApiInterface::create(
    PlatformContextPtr ctx,
    const std::string& suid,
    const task_info& task)
{
    using InfoRequestQuery = query::RequestQuery<popid_t, handlers::api_new_popid>;
    auto req = std::make_shared<InfoRequestQuery>(ctx, "api_create");

    req->addArgs(task.server, task.port, task.login);
    bool isOauth = !task.oauth_refresh_token.empty();
    if (isOauth)
    {
        req->addArg(task.oauth_refresh_token);
    }
    else
    {
        req->addArg(task.password);
    }

    req->addArgs(
        (task.use_ssl ? "Y" : "N"),
        task.email,
        (task.leave_msgs ? "1" : "0"),
        task.abook_sync_state);

    if (task.mark_archive_read)
    {
        req->addArg(boost::posix_time::second_clock::local_time());
    }
    else
    {
        req->addArg(DATE_UNREAD_ARCHIVE);
    }

    req->addArgs(
        suid,
        (task.use_imap ? "1" : "0"),
        task.root_folder,
        (isOauth ? "TRUE" : "FALSE"),
        "FALSE",
        task.label_id);

    return dispatcher->run_on_any(req, request_target::master);
}

future_void_t ApiInterface::remove(PlatformContextPtr ctx, const std::string& suid, popid_t popid)
{
    auto req = std::make_shared<query::ExecuteQuery>(ctx, "api_delete");

    req->addArgs(suid, popid);
    return dispatcher->run(req, popid, request_target::master);
}

future_void_t ApiInterface::edit(
    PlatformContextPtr ctx,
    const std::string& suid,
    const task_info& task)
{
    bool isOauth = !task.oauth_refresh_token.empty();
    bool hasAuth = task.password.size() || isOauth;
    std::string strRequest = (hasAuth ? "api_edit" : "api_edit_nopasswd");
    auto req = std::make_shared<query::ExecuteQuery>(ctx, strRequest);

    req->addArgs(task.server, task.port, task.login);

    if (hasAuth)
    {
        req->addArg(isOauth ? task.oauth_refresh_token : task.password);
    }

    req->addArgs(
        (task.use_ssl ? "Y" : "N"),
        task.email,
        (task.leave_msgs ? "1" : "0"),
        boost::lexical_cast<std::string>(task.use_imap),
        task.root_folder,
        suid,
        task.popid);

    if (hasAuth)
    {
        req->addArg(isOauth ? "TRUE" : "FALSE");
    }

    req->addArgs("FALSE", task.label_id);

    return dispatcher->run(req, task.popid, request_target::master);
}

future_void_t ApiInterface::resetStatus(
    PlatformContextPtr ctx,
    const std::string& /* suid */,
    const task_info& task)
{
    auto req = std::make_shared<query::ExecuteQuery>(ctx, "rpop_update");
    req->addArgs(
        0, // session duration
        task.last_connect,
        task.last_msg_count,
        "ok", // error status
        0,    // bad retries
        task.is_on,
        task.abook_sync_state,
        task.validated,
        task.uidl_hash,
        "", // last server answer
        task.popid);

    return dispatcher->run(req, task.popid, request_target::master);
}

future_void_t ApiInterface::enableCollector(
    PlatformContextPtr ctx,
    const std::string& suid,
    popid_t popid,
    bool on)
{
    auto req = std::make_shared<query::ExecuteQuery>(ctx, "api_enable");

    req->addArgs((on ? "1" : "0"), suid, popid);
    return dispatcher->run(req, popid, request_target::master);
}

future_void_t ApiInterface::enableAbook(
    PlatformContextPtr ctx,
    const std::string& suid,
    const popid_t& popid)
{
    auto req = std::make_shared<query::ExecuteQuery>(ctx, "api_enable");

    req->addArgs(suid, popid);
    return dispatcher->run(req, popid, request_target::master);
}

future_string ApiInterface::getCollectorOwner(PlatformContextPtr ctx, popid_t popid)
{
    using OwnerRequestQuery = query::RequestQuery<std::string, handlers::RpopOwnerHandler>;
    auto req = std::make_shared<OwnerRequestQuery>(ctx, "get_rpop_owner");

    req->addArgs(popid);
    return dispatcher->run(req, popid, request_target::master);
}

FutureStatusData ApiInterface::status(PlatformContextPtr ctx, popid_t popid)
{
    using StatusRequestQuery = query::RequestQuery<StatusData, handlers::StatusHandler>;
    auto req = std::make_shared<StatusRequestQuery>(ctx, "get_collect_status");

    req->addArgs(popid);
    return dispatcher->run(req, popid, request_target::try_replica);
}

FutureStatusData ApiInterface::httpStatus(PlatformContextPtr ctx, popid_t popid)
{
    using StatusRequestQuery = query::RequestQuery<StatusData, handlers::HttpStatusHandler>;
    auto req = std::make_shared<StatusRequestQuery>(ctx, "get_http_collect_status");

    req->addArgs(popid);
    return dispatcher->run(req, popid, request_target::try_replica);
}

future_imap_folders ApiInterface::loadImapFolders(PlatformContextPtr ctx, popid_t popid)
{
    using ImapFoldersRequestQuery = query::RequestQuery<imap_folders_ptr, handlers::imap_folders>;
    auto req = std::make_shared<ImapFoldersRequestQuery>(ctx, "imap_folders_list");

    req->addArgs(popid);
    return dispatcher->run(req, popid, request_target::try_replica);
}

} // namespace yrpopper::db
