#include <db/imap_collector_interface.h>
#include <db/query/request_query.h>
#include <db/query/execute_query.h>

#include <db/handlers/imap_folders.h>
#include <db/handlers/new_folderid.h>
#include <db/handlers/imap_uidl_list.h>

#include <yplatform/find.h>

namespace yrpopper::db {

ImapCollectorInterface::ImapCollectorInterface()
{
    dispatcher = yplatform::find<query_dispatcher>("query_dispatcher");
}

future_imap_folders ImapCollectorInterface::loadImapFolders(PlatformContextPtr ctx, popid_t popid)
{
    using ImapFoldersRequestQuery = query::RequestQuery<imap_folders_ptr, handlers::imap_folders>;
    auto req = std::make_shared<ImapFoldersRequestQuery>(ctx, "imap_folders_list");

    req->addArgs(popid);
    return dispatcher->run(req, popid, request_target::try_replica);
}

future_uint64_t ImapCollectorInterface::appendImapFolder(
    PlatformContextPtr ctx,
    popid_t popid,
    imap_folder_ptr folder)
{
    using AppendFolderRequestQuery = query::RequestQuery<uint64_t, handlers::new_folderid>;
    auto req = std::make_shared<AppendFolderRequestQuery>(ctx, "append_imap_folder");

    req->addArgs(popid, folder->name, folder->uidvalidity, folder->nextuid, folder->exists);
    return dispatcher->run(req, popid, request_target::master);
}

future_void_t ImapCollectorInterface::deleteImapFolders(
    PlatformContextPtr ctx,
    popid_t popid,
    imap_folders_ptr folders)
{
    auto req = std::make_shared<query::ExecuteQuery>(ctx, "delete_imap_folders");

    std::vector<int64_t> folderIds;
    for (auto folder : *folders)
    {
        folderIds.push_back(folder.second->folder_id);
    }
    req->addArgs(popid, std::move(folderIds));
    return dispatcher->run(req, popid, request_target::master);
}

future_void_t ImapCollectorInterface::editImapFolder(PlatformContextPtr ctx, imap_folder_ptr folder)
{
    auto req = std::make_shared<query::ExecuteQuery>(ctx, "edit_imap_folder");

    req->addArgs(
        folder->folder_id,
        folder->uidvalidity,
        folder->nextuid,
        folder->exists,
        (folder->inited ? "TRUE" : "FALSE"));
    return dispatcher->run(req, folder->folder_id, request_target::master);
}

future_imap_uidl_map ImapCollectorInterface::loadImapMessages(
    PlatformContextPtr ctx,
    uint64_t folderId,
    const std::string& state,
    uint32_t count)
{
    using LoadMessagesRequestQuery =
        query::RequestQuery<imap_uidl_map_ptr, handlers::imap_uidl_list>;
    auto req = std::make_shared<LoadMessagesRequestQuery>(ctx, "imap_messages_list");

    req->addArgs(folderId, state, count);
    return dispatcher->run(req, folderId, request_target::try_replica);
}

future_void_t ImapCollectorInterface::addInitialImapMessages(
    PlatformContextPtr ctx,
    uint64_t folderId,
    std::vector<int64_t>&& uidls)
{
    auto req = std::make_shared<query::ExecuteQuery>(ctx, "add_initial_imap_messages");

    req->addArgs(folderId, std::move(uidls));
    return dispatcher->run(req, folderId, request_target::master);
}

future_void_t ImapCollectorInterface::editImapMessage(
    PlatformContextPtr ctx,
    uint64_t folderId,
    uint32_t uidl,
    const std::string& newState)
{
    auto req = std::make_shared<query::ExecuteQuery>(ctx, "edit_imap_message");

    req->addArgs(folderId, uidl, newState);
    return dispatcher->run(req, folderId, request_target::master);
}

} // namespace yrpopper::db
