#pragma once

#include <db/query_dispatcher/settings.h>
#include <db/query/request_query.h>
#include <db/query/execute_query.h>
#include <common/random_uint_generator.h>
#include <ymod_pq/call.h>
#include <yplatform/module.h>
#include <yplatform/future/multi_future.hpp>
#include <boost/shared_ptr.hpp>
#include <algorithm>
#include <memory>
#include <vector>

namespace yrpopper { namespace db {

namespace detail {

template <typename FutureCollection>
inline auto merge_query_results(const FutureCollection& futures)
{
    using future_type = typename FutureCollection::value_type;
    using result_container_ptr_type = typename future_type::value_type;
    using result_container_type = typename result_container_ptr_type::element_type;

    try
    {
        auto result = boost::make_shared<result_container_type>();
        for (auto&& future : futures)
        {
            auto&& query_result = future.get();
            for (auto it = query_result->begin(); it != query_result->end(); ++it)
            {
                result->insert(result->end(), std::move(*it));
            }
        }
        return result;
    }
    catch (const std::exception& ex)
    {
        YLOG_G(error) << "got exception while merging db query results : " << ex.what();
        throw;
    }
}
} // namespace detail

using request_target = enum ymod_pq::request_target;

template <typename RandomUintGen>
class query_dispatcher_impl
    : public yplatform::module
    , private boost::noncopyable
{
public:
    explicit query_dispatcher_impl(const settings& settings) : settings_(settings)
    {
    }

    explicit query_dispatcher_impl(const yplatform::ptree& conf)
        : query_dispatcher_impl(settings(conf))
    {
    }

    template <typename Query>
    auto run(std::shared_ptr<Query> query, sharding_key_t sharding_key, request_target target) const
    {
        auto conninfo = get_shard_conninfo(sharding_key);
        return query->run_on_shard(conninfo, target);
    }

    template <typename Query>
    auto run_on_any(std::shared_ptr<Query> query, request_target target) const
    {
        auto conninfo = get_random_shard_conninfo();
        return query->run_on_shard(conninfo, target);
    }

    template <typename Query>
    auto run_on_all(std::shared_ptr<Query> query, request_target target)
    {
        using result_type = typename Query::ResType;
        using result_promise = yplatform::future::promise<result_type>;
        using result_future = yplatform::future::future<result_type>;

        std::vector<result_future> query_futures;

        for (auto&& shard : settings_.shards)
        {
            auto query_future = query->run_on_shard(shard.master_conninfo, target);
            query_futures.push_back(std::move(query_future));
        }

        auto future_multi = future_multi_and(query_futures);
        return future_multi.then([query_futures = std::move(query_futures)](auto future) mutable {
            if (future.has_exception())
            {
                std::rethrow_exception(future.get_exception());
            }
            return detail::merge_query_results(query_futures);
        });
    }

    auto run_on_all(std::shared_ptr<query::ExecuteQuery> query, request_target target)
    {
        promise_void_t prom;
        std::vector<future_void_t> query_futures;

        for (auto&& shard : settings_.shards)
        {
            auto query_future = query->run_on_shard(shard.master_conninfo, target);
            query_futures.push_back(std::move(query_future));
        }

        auto future_multi = future_multi_and(query_futures);
        return future_multi.then([prom](auto future) mutable {
            if (future.has_exception())
            {
                std::rethrow_exception(future.get_exception());
            }
            return VoidResult{};
        });
    }

private:
    const std::string& get_shard_conninfo(sharding_key_t sharding_key) const
    {
        auto&& shards = settings_.shards;
        auto shard_it = std::find_if(shards.begin(), shards.end(), [sharding_key](const shard& s) {
            return s.start_key <= sharding_key && sharding_key <= s.end_key;
        });
        if (shard_it == shards.end())
        {
            throw std::runtime_error("failed to get shard for key " + std::to_string(sharding_key));
        }
        return shard_it->master_conninfo;
    }

    const std::string& get_random_shard_conninfo() const
    {
        auto&& shards = settings_.shards;
        auto random_index = random_uint_generator_(0, shards.size() - 1);
        return shards[random_index].master_conninfo;
    }

    settings settings_;
    RandomUintGen random_uint_generator_;
};

using query_dispatcher = query_dispatcher_impl<yrpopper::random_uint_generator>;
using query_dispatcher_ptr = boost::shared_ptr<query_dispatcher>;

} // namespace db
} // namespace yrpopper
