#pragma once

#include <yplatform/ptree.h>
#include <cstdint>
#include <vector>
#include <string>

namespace yrpopper { namespace db {

using sharding_key_t = uint64_t;

struct shard
{
    std::string master_conninfo;
    sharding_key_t start_key;
    sharding_key_t end_key;
};

using shard_list = std::vector<shard>;

struct settings
{
    settings() = default;

    settings(const yplatform::ptree& conf)
    {
        shard shard;
        sharding_key_t start_key = 1;
        auto shards_conf_range = conf.equal_range("shards");
        for (auto it = shards_conf_range.first; it != shards_conf_range.second; ++it)
        {
            shard.start_key = start_key;
            shard.end_key = it->second.get<sharding_key_t>("max_key");
            start_key = shard.end_key + 1;
            if (shard.start_key > shard.end_key)
            {
                throw std::runtime_error(
                    "shard start key " + std::to_string(shard.start_key) +
                    " is greater than end key " + std::to_string(shard.end_key));
            }
            load_conninfo(it->second.get_child("conninfo"), shard);
            shards.emplace_back(std::move(shard));
        }
        if (shards.empty())
        {
            throw std::runtime_error(
                "`query_dispatcher` configuration must contain at least one shard");
        }
    }

    void load_conninfo(const yplatform::ptree& conf, shard& shard)
    {
        std::string hosts;
        auto hosts_range = conf.equal_range("hosts");
        for (auto it = hosts_range.first; it != hosts_range.second; ++it)
        {
            hosts.append(it->second.data() + ',');
        }
        if (hosts.empty())
        {
            throw std::runtime_error("shard conninfo should not be empty");
        }
        // Cut off the trailing ','.
        hosts.resize(hosts.size() - 1);
        hosts = "host=" + hosts;
        std::string params = conf.get<std::string>("params", "");
        std::string master_params = params + " " + conf.get<std::string>("master_params", "");
        shard.master_conninfo = hosts + " " + master_params;
    }

    shard_list shards;
};

} // namespace db
} // namespace yrpopper
