#include <db/scheduler_interface.h>
#include <db/query/request_query.h>
#include <db/query/execute_query.h>
#include <db/handlers/chunk_index.h>
#include <db/handlers/chunk_request.h>
#include <db/handlers/rpop_list.h>

namespace yrpopper::db {

SchedulerInterface::SchedulerInterface()
{
    dispatcher = yplatform::find<query_dispatcher>("query_dispatcher");
}

future_void_t SchedulerInterface::updateChunks(
    PlatformContextPtr ctx,
    const std::string& owner_name)
{
    auto req = std::make_shared<query::ExecuteQuery>(ctx, "chunks_update");

    req->addArg(owner_name);
    return dispatcher->run_on_all(req, request_target::master);
}

future_chunk_requester_dict SchedulerInterface::listChunks(
    PlatformContextPtr ctx,
    const std::string& owner_name)
{
    using ChunkRequestersRequestQuery =
        query::RequestQuery<chunk_requester_dict_ptr, handlers::chunk_index>;
    auto req = std::make_shared<ChunkRequestersRequestQuery>(ctx, "chunk_list");

    req->addArg(owner_name);
    return dispatcher->run_on_all(req, request_target::master);
}

future_uint64_t SchedulerInterface::getNewChunk(
    PlatformContextPtr ctx,
    const std::string& owner_name)
{
    using ChunkRequestQuery = query::RequestQuery<uint64_t, handlers::chunk_request>;
    auto req = std::make_shared<ChunkRequestQuery>(ctx, "get_new_chunk");

    req->addArg(owner_name);
    return dispatcher->run_on_any(req, request_target::master);
}

future_void_t SchedulerInterface::releaseChunk(
    PlatformContextPtr ctx,
    const std::string& owner_name,
    uint64_t chunkId)
{
    auto req = std::make_shared<query::ExecuteQuery>(ctx, "chunk_release");

    req->addArgs(owner_name, chunkId);
    return dispatcher->run(req, chunkId, request_target::master);
}

future_void_t SchedulerInterface::updateTask(
    PlatformContextPtr ctx,
    const task_info& task,
    const std::string& server_response)
{
    auto req = std::make_shared<query::ExecuteQuery>(ctx, "rpop_update");

    req->addArgs(
        task.session_duration,
        task.last_connect,
        task.last_msg_count,
        task.error_status,
        task.bad_retries,
        task.is_on,
        task.abook_sync_state,
        task.validated,
        task.uidl_hash,
        server_response,
        task.popid);
    return dispatcher->run(req, task.popid, request_target::master);
}

future_task_ptr_list SchedulerInterface::getTaskList(
    PlatformContextPtr ctx,
    const std::string& owner_name)
{
    using TaskListRequestQuery = query::RequestQuery<task_ptr_list_ptr, handlers::rpop_list>;
    auto req = std::make_shared<TaskListRequestQuery>(ctx, "rpop_list");

    req->addArg(owner_name);
    return dispatcher->run_on_all(req, request_target::try_replica);
}

} // namespace yrpopper::db
