#pragma once

#include <yplatform/log.h>

#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/classification.hpp>

#include <fstream>
#include <mutex>
#include <string>
#include <unordered_map>

namespace yrpopper { namespace oauth { namespace hack  {

class TokenCache
{
public:
    using Tokens = std::unordered_map<std::string, std::string>;

    TokenCache() = default;

    TokenCache(const std::string& path)
    {
        reload(path);
    }

    void reload(const std::string& path)
    {
        try
        {
            auto tokens = reload_tokens(path);
            YLOG_G(info) << "TokenCache loaded oauth access tokens: " << tokens.size();

            std::lock_guard<std::mutex> lock(mutex);
            std::swap(access_tokens, tokens);
        }
        catch(const std::exception& e)
        {
            YLOG_G(error) << "TokenCache reload error: " << e.what();
            return;
        }
    }

    std::string getAccessToken(const std::string& login)
    {
        std::lock_guard<std::mutex> lock(mutex);
        auto it = access_tokens.find(login);
        if (it == access_tokens.end())
        {
            throw std::runtime_error("no access token");
        }
        YLOG_G(debug) << "TokenCache found token for " << login;
        return it->second;
    }

private:

    Tokens reload_tokens(const std::string& path)
    {
        auto tokens = read_from_file(path);
        if (tokens.empty())
        {
            throw std::runtime_error("empty file");
        }
        return tokens;
    }

    Tokens read_from_file(const std::string& path)
    {
        Tokens tokens;
        std::ifstream file(path.c_str());
        if (!file.is_open())
        {
            throw std::runtime_error("failed to open file: " + path);
        }

        std::string line;
        std::vector<std::string> parts;
        while (std::getline(file, line))
        {
            boost::split(parts, line, boost::is_any_of(" "));
            if (parts.size() < 2)
            {
                throw std::runtime_error("failed to parse access tokens file: " + path);
            }
            std::string email = parts[0];
            std::string access_token = parts[1];
            tokens[email] = access_token;
        }
        return tokens;
    }

    std::mutex mutex;
    Tokens access_tokens;
};

}}}
