#include <algorithm>
#include <oauth/oauth_impl.h>

#include <boost/property_tree/ptree.hpp>
#include <boost/property_tree/xml_parser.hpp>

#include <common/util.h>

using namespace std;

namespace yrpopper::oauth {

void OauthServiceImpl::readSettings(const yplatform::ptree& cfg)
{
    settings.servers.clear();

    const yplatform::ptree& servers = cfg.get_child("servers");
    for (auto& i : servers)
    {
        if (i.first == "<xmlattr>") continue;
        if (i.first != "server")
        {
            throw std::runtime_error(
                "wrong settings xml in oauth module, <servers> contains not <server>");
        }

        auto address = i.second.get<string>("<xmlattr>.address");
        auto id = i.second.get<string>("<xmlattr>.id");
        settings.servers.insert(std::pair<string, string>(address, id));
    }

    settings.host = cfg.get<string>("host");
    settings.retries = cfg.get<uint32_t>("retries", 3);
    settings.socialApiTvmService = cfg.get<string>("social_api_tvm_service");
    settings.socialProxyTvmService = cfg.get<string>("social_proxy_tvm_service");
}

void OauthServiceImpl::init(const yplatform::ptree& xml)
{
    readSettings(xml);
}

void OauthServiceImpl::reload(const yplatform::ptree& xml)
{
    readSettings(xml);
}

FutureRefreshTokenData OauthServiceImpl::getRefreshToken(
    yplatform::task_context_ptr ctx,
    const std::string& taskId)
{
    auto tokenLoader = std::make_shared<TokenLoader>(settings, ctx);
    return tokenLoader->getRefreshToken(taskId);
}

FutureString OauthServiceImpl::getAccessToken(
    yplatform::task_context_ptr ctx,
    const std::string& server,
    const std::string& refreshToken)
{
    auto tokenLoader = std::make_shared<TokenLoader>(settings, ctx);
    return tokenLoader->getAccessToken(server, refreshToken);
}

} // namespace yrpopper::oauth

#include <yplatform/module_registration.h>

DEFINE_SERVICE_OBJECT(yrpopper::oauth::OauthServiceImpl)
