#pragma once

#include <oauth/oauth.h>
#include <common/data_source.h>
#include <common/http_loader.h>
#include <ymod_tvm/module.h>

#include <butil/butil.h>

namespace yrpopper { namespace oauth {

inline std::string makeTvmHeader(const std::string& serviceName)
{
    auto tvm = yplatform::find<ymod_tvm::tvm2_module>("tvm");
    std::string serviceTicket;
    auto err = tvm->get_service_ticket(serviceName, serviceTicket);
    if (err)
        throw std::runtime_error(
            "can't get serivce ticket for " + serviceName + "(" + err.message() + ")");
    return "X-Ya-Service-Ticket: " + serviceTicket + "\r\n";
}

//-----------------------------------------------------------------------------
// Refresh token loader: task_id -> refresh_token

class RefreshTokenLoader : public JsonHttpLoader<HttpGet>
{
public:
    RefreshTokenLoader(task_context_ptr context, const Settings& settings)
        : JsonHttpLoader(context, settings.host, settings.retries)
        , socialApiTvmService(settings.socialApiTvmService)
    {
    }

    virtual std::string getRequestData()
    {
        return request;
    }

    std::string getHeaders() override
    {
        return makeTvmHeader(socialApiTvmService);
    }

    FutureRefreshTokenData loadToken(const std::string& taskId)
    {
        request = "/api/task/" + taskId + "?consumer=mail-collectors-ext";
        load();
        return tokenPromise;
    }

    virtual void handleError(const std::string& errorMessage)
    {
        auto strData = data.str();
        if (strData.empty())
        {
            tokenPromise.set_exception(std::runtime_error(errorMessage));
            return;
        }

        std::string errDescription;
        try
        {
            boost::property_tree::ptree errorPtree;
            boost::property_tree::read_json(this->data.seekg(std::ios_base::beg), errorPtree);
            errDescription = getOauthError(errorPtree, this->data.str());
        }
        catch (...)
        {
            errDescription = std::move(data.str());
        }

        tokenPromise.set_exception(std::runtime_error(errorMessage + ": " + errDescription));
    }

protected:
    virtual void handleJson(boost::property_tree::ptree&& oauthPtree)
    {
        auto refreshToken = oauthPtree.get("token.refresh", "");
        auto profileEmail = oauthPtree.get("profile.email", "");
        auto provider = oauthPtree.get("profile.provider.name", "");

        if (refreshToken.empty())
        {
            tokenPromise.set_exception(
                incorrect_data_exception(getOauthError(oauthPtree, "empty refresh token")));
        }
        else if (profileEmail.find('@') == string::npos)
        {
            tokenPromise.set_exception(
                incorrect_data_exception(getOauthError(oauthPtree, "wrong email")));
        }
        else
        {
            tokenPromise.set({ refreshToken, profileEmail, provider });
        }
    }

    std::string getOauthError(
        const boost::property_tree::ptree& oauthPtree,
        const string& description)
    {
        string error = oauthPtree.get("error.name", "general error");
        string desc = oauthPtree.get("error.description", description);
        return "error: " + error + ", desc: " + desc;
    }

protected:
    std::string socialApiTvmService;
    std::string request;
    PromiseRefreshTokenData tokenPromise;
};

//-----------------------------------------------------------------------------
// Access token loader: refresh_token -> access_token

class AccessTokenLoader : public JsonHttpLoader<HttpPost>
{
public:
    AccessTokenLoader(task_context_ptr context, const Settings& settings)
        : JsonHttpLoader(context, settings.host, settings.retries)
        , socialProxyTvmService(settings.socialProxyTvmService)
    {
    }

    FutureString loadToken(const std::string& serverId, const std::string& refreshToken)
    {
        request = "/proxy2/application/" + serverId + "/refresh_token?consumer=mail-collectors-ext";
        post = boost::make_shared<string>("refresh_token=" + refreshToken);
        load();
        return tokenPromise;
    }

    virtual std::string getRequestData()
    {
        return request;
    }
    virtual http_string_ptr getPostData()
    {
        return post;
    }

    std::string getHeaders() override
    {
        return makeTvmHeader(socialProxyTvmService);
    }

    virtual void handleError(const std::string& errorMessage)
    {
        auto strData = data.str();
        if (strData.empty())
        {
            tokenPromise.set_exception(std::runtime_error(errorMessage));
            return;
        }

        std::string errDescription;
        try
        {
            boost::property_tree::ptree errorPtree;
            boost::property_tree::read_json(this->data.seekg(std::ios_base::beg), errorPtree);
            errDescription = getOauthError(errorPtree, this->data.str());
        }
        catch (...)
        {
            errDescription = std::move(data.str());
        }

        tokenPromise.set_exception(std::runtime_error(errorMessage + ": " + errDescription));
    }

protected:
    virtual void handleJson(boost::property_tree::ptree&& oauthPtree)
    {
        auto accessToken = oauthPtree.get("result.access_token", "");
        if (accessToken.empty())
        {
            tokenPromise.set_exception(
                incorrect_data_exception(getOauthError(oauthPtree, "no access token")));
        }
        else
        {
            tokenPromise.set(accessToken);
        }
    }

    std::string getOauthError(
        const boost::property_tree::ptree& oauthPtree,
        const string& description)
    {
        string error = oauthPtree.get("task.reason.code", "general error");
        string desc = oauthPtree.get("task.reason.description", description);
        return "error: \"" + error + "\" desc: \"" + desc + "\"";
    }

protected:
    PromiseString tokenPromise;
    std::string socialProxyTvmService;
    std::string request;
    http_string_ptr post;
};

//-----------------------------------------------------------------------------
// Combined token loader: task_id -> refresh_token -> access_token

struct TokenLoader : public std::enable_shared_from_this<TokenLoader>
{
    TokenLoader(const Settings& settings, yplatform::task_context_ptr ctx)
        : settings(settings), context(ctx)
    {
    }

    FutureRefreshTokenData getRefreshToken(const std::string& taskId)
    {
        auto refreshTokenLoader = boost::make_shared<RefreshTokenLoader>(context, settings);
        return refreshTokenLoader->loadToken(taskId);
    }

    FutureString getAccessToken(const std::string& server, const std::string& refreshToken)
    {
        auto srv = settings.servers.find(server);
        if (srv == settings.servers.end())
        {
            PromiseString errorPromise;
            errorPromise.set_exception(
                std::runtime_error("No known oauth providers for server: " + server));
            return errorPromise;
        }

        auto accessTokenLoader = boost::make_shared<AccessTokenLoader>(context, settings);
        return accessTokenLoader->loadToken(srv->second, refreshToken);
    }

protected:
    const Settings& settings;
    yplatform::task_context_ptr context;
};

} // namespace oauth
} // namespace yrpopper
