#include <boost/format.hpp>
#include <processor/handlers/error.h>
#include <processor/handlers/abook_sync.h>

namespace yrpopper::processor::handlers {

namespace {

void proxy_on_start_element(void* user_data, const XML_Char* name, const XML_Char** atts)
{
    (reinterpret_cast<abook_sync*>(user_data))->on_start_element(user_data, name, atts);
}

void proxy_on_end_element(void* user_data, const XML_Char* name)
{
    (reinterpret_cast<abook_sync*>(user_data))->on_end_element(user_data, name);
}

void proxy_on_characters(void* user_data, const XML_Char* s, int len)
{
    (reinterpret_cast<abook_sync*>(user_data))->on_characters(user_data, s, len);
}

int proxy_unknown_encoding(void* /* userData */, const XML_Char* /* name */, XML_Encoding* info)
{
    for (int i = 0; i < 256; ++i)
        info->map[i] = i;
    info->data = NULL;
    info->convert = NULL;
    info->release = NULL;
    return XML_STATUS_OK;
}
}

abook_sync::abook_sync(rpop_context_ptr ctx)
    : confirm_tag_(false)
    , in_error_tag_(false)
    , rec_count_(0)
    , m_status_error(false)
    , m_error(false)
    , ctx_(ctx)
{
    parser_ = XML_ParserCreate(0);
    XML_SetUserData(parser_, this);
    XML_SetElementHandler(parser_, &proxy_on_start_element, &proxy_on_end_element);
    XML_SetCharacterDataHandler(parser_, &proxy_on_characters);
    XML_SetUnknownEncodingHandler(parser_, &proxy_unknown_encoding, 0);
}

ymod_http_client::handler_version abook_sync::version() const
{
    return ymod_http_client::handler_version_already_parse_body;
}

void abook_sync::set_code(int code, const std::string& description)
{
    switch (code)
    {
    case 400:
        m_status_error = true;
    case 200:
        return;
    }
    error_info_ +=
        str(boost::format("server returned bad status: code=%d, description='%s'") % code %
            description);
    m_error = true;
}

void abook_sync::handle_data(const char* data, unsigned long long size)
{
    if (!m_error)
    {
        if (!XML_Parse(parser_, data, size, 0))
        {
            XML_Error error_code = XML_GetErrorCode(parser_);
            if (!error_info_.empty()) error_info_ += ". ";
            error_info_ +=
                str(boost::format("xml abook_sync parser error: code=%d description='%s'") %
                    error_code % XML_ErrorString(error_code));
        }
        if (error_info_.empty() || in_error_tag_) return;
    }
    THROW_HANDLER_ERROR(ctx_, error_info_);
}

void abook_sync::handle_data_end()
{
    if (XML_Parse(parser_, 0, 0, 1)) return;
    XML_Error err_code = XML_GetErrorCode(parser_);
    string err_desc = XML_ErrorString(err_code);
    std::stringstream error_stream;
    error_stream << "xml abook_sync parser error: code=" << err_code << " description='" << err_desc
                 << "'";
    if (error_info_.empty()) error_info_ = error_stream.str();
    else
        error_info_ += ". " + error_stream.str();
    THROW_HANDLER_ERROR(ctx_, error_info_);
}

void abook_sync::on_start_element(
    void* /* user_data */,
    const XML_Char* name,
    const XML_Char** /* atts */)
{
    confirm_tag_ = (confirm_tag_ || (!m_status_error && strcmp(name, "imported") == 0));
    if (!confirm_tag_ && strcmp(name, "error") == 0)
    {
        error_info_ += "abook_sync return error: ";
        in_error_tag_ = true;
    }
}

void abook_sync::on_end_element(void* /* user_data */, const XML_Char* name)
{
    if (confirm_tag_ && strcmp(name, "imported") == 0) rec_count_ = atoi(m_buffer.c_str());
    else if (!confirm_tag_ && strcmp(name, "error") == 0)
        in_error_tag_ = false;
}

void abook_sync::on_characters(void* /* user_data */, const XML_Char* data, int len)
{
    if (in_error_tag_) error_info_.append(data, len);
    else if (confirm_tag_)
        m_buffer.append(data, len);
}

} // namespace yrpopper::processor::handlers
