#include <processor/handlers/error.h>
#include <processor/handlers/validate.h>

namespace yrpopper::processor::handlers {

namespace {

void proxy_on_start_element(void* user_data, const XML_Char* name, const XML_Char** atts)
{
    (reinterpret_cast<validate*>(user_data))->on_start_element(user_data, name, atts);
}

void proxy_on_end_element(void* user_data, const XML_Char* name)
{
    (reinterpret_cast<validate*>(user_data))->on_end_element(user_data, name);
}

void proxy_on_characters(void* user_data, const XML_Char* s, int len)
{
    (reinterpret_cast<validate*>(user_data))->on_characters(user_data, s, len);
}

int proxy_unknown_encoding(void* /* userData */, const XML_Char* /* name */, XML_Encoding* info)
{
    for (int i = 0; i < 256; ++i)
        info->map[i] = i;
    info->data = NULL;
    info->convert = NULL;
    info->release = NULL;
    return XML_STATUS_OK;
}
}

validate::validate(rpop_context_ptr ctx) : confirm_tag_(false), in_error_tag_(false), ctx_(ctx)
{
    parser_ = XML_ParserCreate(0);
    XML_SetUserData(parser_, this);
    XML_SetElementHandler(parser_, &proxy_on_start_element, &proxy_on_end_element);
    XML_SetCharacterDataHandler(parser_, &proxy_on_characters);
    XML_SetUnknownEncodingHandler(parser_, &proxy_unknown_encoding, 0);
}

void validate::handle_data(const char* data, unsigned long long size)
{
    XML_Status result = XML_Parse(parser_, data, size, 0);
    if (!result)
    {
        XML_Error err_code = XML_GetErrorCode(parser_);
        string err_desc = XML_ErrorString(err_code);
        std::stringstream error_stream;
        error_stream << "xml validate parser error: code=" << err_code << " description='"
                     << err_desc << "'";
        if (error_info_.empty()) error_info_ = error_stream.str();
        else
            error_info_ += ". " + error_stream.str();
    }
    if (error_info_.empty() || in_error_tag_) return;
    error_info_ += error_description_;
    THROW_HANDLER_ERROR(ctx_, error_info_);
}

void validate::handle_data_end()
{
    if (XML_Parse(parser_, 0, 0, 1)) return;
    XML_Error err_code = XML_GetErrorCode(parser_);
    string err_desc = XML_ErrorString(err_code);
    std::stringstream error_stream;
    error_stream << "xml validate parser error: code=" << err_code << " description='" << err_desc
                 << "'";
    if (error_info_.empty()) error_info_ = error_stream.str();
    else
        error_info_ += ". " + error_stream.str();
    THROW_HANDLER_ERROR(ctx_, error_info_);
}

void validate::on_start_element(void* /* user_data */, const XML_Char* name, const XML_Char** atts)
{
    confirm_tag_ = !strcmp(name, "validator-rpop-added");
    in_error_tag_ = false;
    if (!confirm_tag_ && strcmp(name, "validator-invalid-argument") == 0)
    {
        std::stringstream error_stream;
        error_stream << "validator return error, params: ";
        const XML_Char** atts_iter = atts;
        for (; *atts_iter; atts_iter += 2)
        {
            if (atts_iter != atts) error_stream << ", ";
            error_stream << string(*atts_iter) << "=" << string(*(atts_iter + 1));
        }
        error_stream << "; reason: ";
        error_info_ += error_stream.str();
        in_error_tag_ = true;
    }
}

void validate::on_end_element(void* /* user_data */, const XML_Char* name)
{
    if (!confirm_tag_ && strcmp(name, "validator-invalid-argument") == 0) in_error_tag_ = false;
}

void validate::on_characters(void* /* user_data */, const XML_Char* data, int len)
{
    if (in_error_tag_) error_description_.append(data, len);
}

} // namespace yrpopper::processor::handlers
