#include <processor/rpop/impl.h>
#include <common/util.h>
#include <yplatform/encoding/url_encode.h>
#include <boost/format.hpp>
#include <boost/filesystem.hpp>
#include <boost/algorithm/string/case_conv.hpp>
#include <boost/smart_ptr/make_shared.hpp>
#include <processor/triggers/email_validator.h>
#include <processor/triggers/abook.h>

#include <ctime>
#include <cstdlib>
#include <cstring>
#include <unistd.h>
#include <sys/stat.h>
#include <functional>

namespace yrpopper::processor {

namespace {

void load_local_mail_from_file(const string& path, string& mail)
{
    std::ifstream rf(path.c_str());
    if (!rf.is_open() || rf.bad()) return;
    int c;
    mail.reserve(10000);
    while ((c = rf.get()) != -1)
        mail += c;
}

void load_mail_from_file(const string& path, settings::email_template& mail)
{
    boost::filesystem::path p(path);
    boost::filesystem::directory_iterator begin(p);
    boost::filesystem::directory_iterator end;
    string email;
    for (; begin != end; ++begin)
    {
        load_local_mail_from_file(begin->path().string(), email);
        mail.localizations.insert(std::make_pair(begin->path().filename().string(), email));
        L_(info) << "load " << begin->path().filename().string() << " localization from "
                 << begin->path().string();
        email.clear();
    }
}

}

rpopprocessor_impl::rpopprocessor_impl()
    : processor()
    , enqueue_timeout_(boost::posix_time::pos_infin)
    , settings_(new settings)
    , sent_counter_("sent")
{
    L_(info) << "rpop_rpopprocessor_impl task instantiated";
    after_trigger_.add_trigger(triggers::trigger_ptr(new triggers::email_validator()));
    after_trigger_.add_trigger(triggers::trigger_ptr(new triggers::abook()));
}

rpopprocessor_impl::~rpopprocessor_impl()
{
    L_(info) << "rpop_rpopprocessor_impl task destroyed";
}

void rpopprocessor_impl::init(const yplatform::ptree& xml)
{
    this->init_pool();
    unsigned nthreads = xml.get("threads", 10);
    unsigned queue_capacity = xml.get("queue_capacity", 10000);
    int enqueue_timeout = xml.get("enqueue_timeout", -1);
    if (enqueue_timeout >= 0)
    {
        enqueue_timeout_ = boost::posix_time::milliseconds(enqueue_timeout);
    }
    boost::optional<const yplatform::ptree&> validator_cfg = xml.get_child_optional("validator");
    if (validator_cfg)
    {
        for (yplatform::ptree::const_iterator i = validator_cfg->begin(),
                                              i_end = validator_cfg->end();
             i != i_end;
             ++i)
        {
            if (i->first == "url")
            {
                settings_->validator_hosts.push_back(i->second.get_value(""));
                L_(info) << "rpop_rpopprocessor_impl: validator "
                         << settings_->validator_hosts.back() << " added";
            }
            else if (i->first == "validate_request")
            {
                settings_->validator_request = i->second.get_value("");
            }
        }
    }

    boost::optional<const yplatform::ptree&> abook_cfg = xml.get_child_optional("abook_sync");
    if (abook_cfg)
    {
        for (yplatform::ptree::const_iterator i = abook_cfg->begin(), i_end = abook_cfg->end();
             i != i_end;
             ++i)
        {
            if (i->first == "url")
            {
                settings_->abook_sync_hosts.push_back(i->second.get_value(""));
                L_(info) << "rpop_rpopprocessor_impl: abook_sync "
                         << settings_->abook_sync_hosts.back() << " added";
            }
        }
    }
    settings_->abook_sync_request = xml.get("abook_sync.request.url", "");
    settings_->abook_sync_request_body = xml.get("abook_sync.request.body", "");
    settings_->enable_abook_sync = (xml.get("abook_sync.enable", "0") != "0");
    settings_->abook_retries_limit = xml.get("abook_sync.retries", settings_->abook_retries_limit);
    {
        string abook_sync_start = xml.get("abook_sync.sync_start", "");
        if (!abook_sync_start.empty())
        {
            settings_->abook_sync_start_date =
                to_unix_time(boost::posix_time::time_from_string(abook_sync_start));
        }
    }
    load_limits(xml);
    settings_->save_path = xml.get("save_path", "");
    settings_->sent_folder_name = xml.get("sent_folder_name", "");
    settings_->enable_validator = (xml.get("validator.enable", "0") != "0");
    settings_->bb_login_service = xml.get("bb_login_service", "");
    settings_->enable_yandex_optimization = (xml.get("yandex_optimization", "") == "on");
    settings_->default_language = xml.get("default_language", "ru");
    load_mail_from_file(xml.get("big_mail_notify_path", ""), settings_->big_mail_notify);
    string (*to_lower_func_c)(const string&, const std::locale&) = &boost::to_lower_copy;
    std::function<string(const string&, const std::locale&)> to_lower_func = to_lower_func_c;
    load_line_file(
        xml.get("remove_headers", ""),
        "system headers",
        boost::bind(
            &insert_string_2_set,
            boost::bind(to_lower_func, _1, std::locale()),
            boost::ref(settings_->remove_headers)));

    after_trigger_.init(settings_);
    this->set_capacity(queue_capacity);
    this->open(nthreads);
}

void rpopprocessor_impl::reload(const yplatform::ptree& xml)
{
    load_limits(xml);
    after_trigger_.reload(settings_);
}

void rpopprocessor_impl::start()
{
}

void rpopprocessor_impl::fini(void)
{
    close();
}

void rpopprocessor_impl::load_limits(const yplatform::ptree& xml)
{
    settings_->save_retries = xml.get("save_retries", 5);
    settings_->max_sent_size = xml.get("max_sent_size", 100 * 1024 * 1024); // 100mb
    settings_->max_mail_size = xml.get("max_mail_size", 41 * 1024 * 1024);
    settings_->max_sent_count = xml.get("max_sent_count", 500);
    settings_->validator_brake.first = xml.get("validator.max_count", 100);
    settings_->validator_brake.second =
        boost::posix_time::seconds(xml.get("validator.interval", 1));
    settings_->validator_brake.first = xml.get("abook_sync.max_count", 100);
    settings_->validator_brake.second =
        boost::posix_time::seconds(xml.get("abook_sync.interval", 1));
    settings_->blackbox_cache_invalidate = xml.get("blackbox_cache_invalidate", 4 * 60 * 60);
    sent_counter_.max_count(xml.get("sent_brake.<xmlattr>.max_count", 300));
    sent_counter_.dimension_interval(
        boost::posix_time::seconds(xml.get("sent_brake.<xmlattr>.interval", 1)));
}

void rpopprocessor_impl::deadline_cancel_hook(rpop_args_ptr args)
{
    args->process_status->session_duration = time(0) - args->process_status->start_time;
    args->prom.set(args->process_status);
}

void rpopprocessor_impl::finish(rpop_args_ptr args)
{
    if (args->task_error)
    {
        return fail_finish(args, args->task_error);
    }
    triggers::future_void_t fres = after_trigger_(args);
    fres.add_callback(boost::bind(
        &rpopprocessor_impl::handle_after_trigger, yplatform::shared_from(this), args, fres));
}

void rpopprocessor_impl::handle_after_trigger(rpop_args_ptr args, future_void_t /* res */)
{
    success_finish(args);
}

void rpopprocessor_impl::success_finish(rpop_args_ptr args)
{
    args->process_status->session_duration = time(0) - args->process_status->start_time;
    args->process_status->error = code::ok;
    args->process_status->bad_retries = 0;
    args->context->success_finished = true;
    args->prom.set(args->process_status);
}

void rpopprocessor_impl::fail_finish(
    rpop_args_ptr args,
    error ec,
    bool inc_bad,
    const string& server_response)
{
    args->process_status->session_duration = time(0) - args->process_status->start_time;
    args->process_status->error = ec;
    args->process_status->server_response = server_response;
    args->process_status->uidl_hash.clear();
    if (inc_bad) ++args->process_status->bad_retries;
    args->context->success_finished = false;
    args->prom.set(args->process_status);
}

} // namespace yrpopper::processor

#include <yplatform/module_registration.h>
DEFINE_SERVICE_OBJECT(yrpopper::processor::rpopprocessor_impl)
