#pragma once

#include <boost/thread.hpp>
#include <boost/property_tree/ptree.hpp>
#include <boost/shared_ptr.hpp>

#include <yplatform/future/future.hpp>
#include <yplatform/repository.h>
#include <yplatform/find.h>
#include <yplatform/log.h>
#include <yplatform/zerocopy/streambuf.h>
#include <yplatform/exception.h>

#include <smtp/module.h>
#include <common/context.h>
#include <common/util.h>

#include <processor/args.h>
#include <processor/processor.h>
#include <processor/settings.h>
#include <processor/triggers/multiplexor.h>
#include <processor/sent_counter.h>
#include <processor/rpop/context.h>
#include <ymod_popclient/errors.h>
#include <ymod_popclient/call.h>
#include <ymod_smtpclient/call.h>
#include <ymod_blackbox/auth.h>
#include <functional>
#include <limits>
#include <map>

namespace yrpopper { namespace processor {

typedef yplatform::future::future<ymod_blackbox::response> auth_response_t;

struct message_info
{
    message_info(const string& uidl)
        : uidl(uidl)
        , last_received_date(std::numeric_limits<time_t>::max())
        , max_received_date(0)
        , send_date(0)
        , is_sender(false)
        , is_loop(false)
        , subject()
        , crlf()
        , is_recovery_email(false)
    {
    }

    string uidl;
    time_t last_received_date;
    time_t max_received_date;
    time_t send_date;
    bool is_sender;
    bool is_loop;
    string subject;
    string crlf;
    bool is_recovery_email;
    string message_id;
};

class rpopprocessor_impl : public processor
{
    typedef std::function<void()> bb_callback_t;

public:
    rpopprocessor_impl();
    virtual ~rpopprocessor_impl();

    void init(const yplatform::ptree& xml);
    void reload(const yplatform::ptree& xml);
    void fini(void);
    void start();

    future_task_status_ptr process(
        rpop_context_ptr context,
        const yplatform::active::ptime& deadline)
    {
        rpop_args_ptr args(new rpop_args_t(context));
        args->dbInterface = db::InterfaceProvider::getLegacyInterface();
        args->process_status->abook_sync_state = args->context->task->abook_sync_state;
        args->process_status->validated = args->context->task->validated;

        if (!this->enqueue_method(
                boost::bind(&rpopprocessor_impl::begin_rpop, this, args),
                boost::bind(&rpopprocessor_impl::deadline_cancel_hook, this, args),
                1,
                deadline))
        {
            args->process_status->session_duration = time(0) - args->process_status->start_time;
            args->process_status->error = code::internal_error;
            args->prom.set(args->process_status);
        }

        return args->prom;
    }

protected:
    void load_limits(const yplatform::ptree& xml);

    typedef ymod_pop_client::message_list_t::const_iterator message_iter;
    void deadline_cancel_hook(rpop_args_ptr args);

    void begin_rpop(const rpop_args_ptr& args);
    void load_uidl_set(const rpop_args_ptr& args);
    void process_message(rpop_args_ptr args, message_iter iter);
    void report_message_and_process_next(
        rpop_args_ptr args,
        message_iter iter,
        const std::string& status,
        const std::string& error,
        const message_info& msg_info);
    void report_message(
        rpop_args_ptr args,
        const std::string& status,
        const std::string& error,
        const message_info& msg_info,
        const std::string& smtp_response = "");
    void process_next_message(rpop_args_ptr args, message_iter iter);
    void process_load_gmail_msg(const rpop_args_ptr& args, message_iter iter);
    void delete_message(rpop_args_ptr args, message_iter iter, bool app_uidl = true);
    void request_message(rpop_args_ptr args, message_iter iter);
    void append_uidl(rpop_args_ptr args, message_iter iter, std::size_t source_size = 0);

    void pop_quit(rpop_args_ptr args);
    bool need_save_uidls(rpop_args_ptr args);
    void save_uidls(rpop_args_ptr args);

    void add_rpop_headers(
        std::ostream& stream,
        const rpop_args_ptr& args,
        const message_info& msg_info);
    std::pair<yplatform::zerocopy::segment, std::size_t> insert_headers(
        rpop_args_ptr args,
        const string& message,
        message_info& msg_info);

    bool save_message(rpop_args_ptr args, yplatform::zerocopy::segment message);

    void precheck_yandex(const rpop_args_ptr& args);

    void handle_yandex_login(const rpop_args_ptr& args, auth_response_t var);

    void handle_yandex_lcn(const rpop_args_ptr& args, future_string res);

    void handle_load_uidls(rpop_args_ptr args, future_uidl_map res); // uidls loaded

    void connect_to_pop3_server(const rpop_args_ptr& args);

    void handle_pop_connect(
        rpop_args_ptr args,
        ymod_pop_client::future_connect_result res); // connect to pop3 server

    void handle_pop_login(
        rpop_args_ptr args,
        ymod_pop_client::future_bool_t res); // login to pop3 server

    void handle_pop_list(
        rpop_args_ptr args,
        ymod_pop_client::future_msg_list_ptr res); // message list loaded

    void handle_pop_message(
        rpop_args_ptr args,
        ymod_pop_client::future_string_ptr res,
        message_iter iter); // pop message loaded

    void handle_gmail_message(
        const rpop_args_ptr& args,
        ymod_pop_client::future_string_ptr res,
        message_iter iter);

    void handle_smtp_message(
        rpop_args_ptr args,
        yplatform::zerocopy::segment message,
        Future<smtp::SMTPResult> res,
        message_iter iter,
        const message_info& info,
        std::size_t source_size); // message sent smtp

    void handle_append_uidl(
        rpop_args_ptr args,
        message_iter iter,
        future_void_t res); // message sent smtp

    void handle_pop_delete(
        rpop_args_ptr args,
        ymod_pop_client::future_bool_t res,
        message_iter iter,
        bool app_uidl); // pop message deleted

    void handle_pop_quit(
        rpop_args_ptr args,
        ymod_pop_client::future_bool_t res); // quit from pop3 server

    void handle_save_uidls(rpop_args_ptr args, future_void_t res); // uidls saved - finish task

    void handle_after_trigger(rpop_args_ptr args, future_void_t res);

    yplatform::active::time_duration enqueue_timeout_;

private:
    template <typename Future>
    bool is_finished(rpop_args_ptr args, error ec, const string& state, const Future& fres);

    virtual const yplatform::active::time_duration get_enqueue_timeout() const
    {
        return enqueue_timeout_;
    }

    std::shared_ptr<rpopprocessor_impl> get_shared_from_this()
    {
        return std::dynamic_pointer_cast<rpopprocessor_impl>(shared_from_this());
    }

    void finish(rpop_args_ptr args);
    void success_finish(rpop_args_ptr args);
    void fail_finish(
        rpop_args_ptr args,
        error ec,
        bool inc_bad = false,
        const string& server_response = "");

    void request_blackbox(rpop_args_ptr args, const bb_callback_t& cb);
    void handle_blackbox(rpop_args_ptr args, auth_response_t var, const bb_callback_t& cb);

    rpop_processor_ctx_ptr get_ctx(yplatform::task_context_ptr context)
    {
        return context->create_module_data<rpop_processor_ctx>("rpop_processor");
    }

    settings_ptr settings_;
    sent_counter sent_counter_;
    triggers::multiplexor after_trigger_;
};

template <typename Future>
bool rpopprocessor_impl::is_finished(
    rpop_args_ptr args,
    error ec,
    const string& state,
    const Future& fres)
{
    if (!fres.has_exception()) return false;
    const string* response = 0;
    try
    {
        fres.get();
    }
    catch (const ymod_pop_client::error& x)
    {
        response = boost::get_error_info<ymod_pop_client::server_response_info>(x);
    }
    catch (const yplatform::exception& x)
    {
        response = boost::get_error_info<ymod_pop_client::server_response_info>(x);
    }
    catch (...)
    {
    }
    if (response)
    {
        YRIMAP_ERROR(args->context) << state << ": status=error, response='" << *response << "' ("
                                    << get_exception_reason(fres) << ")";
        this->fail_finish(args, ec, true, *response);
    }
    else
    {
        YRIMAP_ERROR(args->context)
            << state << ": status=error (" << get_exception_reason(fres) << ")";
        this->fail_finish(args, ec, true);
    }
    return true;
}

inline bool isGmailEmail(const rpop_args_ptr& args)
{
    return args->context->task->server == "pop.gmail.com";
}

inline bool isYandexEmail(const rpop_args_ptr& args)
{
    return args->context->task->server == "pop.yandex.ru";
}

} // namespace processor
} // namespace yrpopper
