#include <common/util.h>
#include <processor/rpop/impl.h>
#include <processor/rfc822date.h>
#include <boost/format.hpp>

#include <yplatform/convert/yconv.h>
#include <yplatform/util.h>
#include <butil/butil.h>
#include <butil/digest.h>
#include <mimeparser/part.h>
#include <mimeparser/rfc2822address.h>
#include <mimeparser/rfc2822date.h>
#include <mimeparser/rfc2047.h>
#include <mimeparser/HeaderParser.h>
#include <boost/algorithm/string/case_conv.hpp>

namespace yrpopper::processor {

namespace {

bool is_space_or_nline(int c)
{
    if (c == ' ' || c == '\r' || c == '\n' || c == '\t') return true;
    return false;
}

// remove leading lines which are not part of rfc822
// some pop3 servers return "From envname date" first line from mbox
// see http://www.qmail.org/qmail-manual-html/man5/mbox.html
string::size_type sanitize_message(const string& msg)
{
    string::size_type offset = 0;
    string::size_type size = msg.size();
    // eat leading"\r\n\t\ "
    while (is_space_or_nline(msg[offset]) && offset < size)
        ++offset;
    string from_ = "from ";
    // if line starts as "From ... "
    if (offset + from_.size() < size)
    {
        if (yplatform::util::iequals(
                boost::make_iterator_range(
                    msg.begin() + offset, msg.begin() + offset + from_.size()),
                boost::make_iterator_range(from_.begin(), from_.end())))
        {
            // move until end of line
            while (msg[offset] != '\r' && msg[offset] != '\n' && offset < size)
                ++offset;
            // eat trailing "\r\n\t\ "
            while (is_space_or_nline(msg[offset]) && offset < size)
                ++offset;
        }
    }
    return offset;
}

template <class ForwardTraversalIterator, class Stream>
class header_handler
{
public:
    typedef ForwardTraversalIterator iterator;
    typedef boost::iterator_range<ForwardTraversalIterator> range;

    header_handler(
        iterator begin,
        Stream& stream,
        const rpop_args_ptr& args,
        const message_info& info,
        const settings_ptr& settings)
        : is_parsed_(false)
        , header_end_(begin)
        , stream_(stream)
        , info_(info)
        , args_(args)
        , settings_(settings)
    {
    }

    bool beginHeader(const iterator& /* it */)
    {
        return true;
    }
    bool endHeader(const iterator& it)
    {
        is_parsed_ = true;
        header_end_ = it;
        return true;
    }
    bool headerField(const range& st, const range& eol);
    bool isParsed() const
    {
        return is_parsed_;
    }
    iterator header_end()
    {
        return header_end_;
    }

    const message_info& info() const
    {
        return info_;
    }

private:
    time_t parse_date(const string& str);
    time_t parse_received(const string& str);
    bool parse_from(const string& from_str);
    bool parse_loop(const string& suid_md5);
    void parse_subject(const string& source);

    bool is_parsed_;
    iterator header_end_;
    Stream& stream_;
    message_info info_;
    rpop_args_ptr args_;
    settings_ptr settings_;
};

template <class ForwardTraversalIterator, class Stream>
time_t header_handler<ForwardTraversalIterator, Stream>::parse_date(const string& str)
{
    std::size_t wh_count = 0;
    while (std::isspace(str[wh_count]))
        ++wh_count;
    if (wh_count)
    {
        rfc2822::rfc2822date date(str.substr(wh_count));
        return date.unixtime();
    }
    else
    {
        rfc2822::rfc2822date date(str);
        return date.unixtime();
    }
}

template <class ForwardTraversalIterator, class Stream>
time_t header_handler<ForwardTraversalIterator, Stream>::parse_received(const string& str)
{
    string::size_type date_start = str.rfind(';');
    if (date_start == str.npos) return 0;

    while (date_start < str.size() && !isalnum(str[date_start]))
        ++date_start;
    if (date_start < str.size()) return parse_date(str.substr(date_start));
    else
        return 0;
}

template <class ForwardTraversalIterator, class Stream>
bool header_handler<ForwardTraversalIterator, Stream>::parse_from(const string& from_str)
{
    rfc2822::rfc2822address addr_list(from_str);
    if (!addr_list.ok()) return false;
    for (rfc2822::rfc2822address::const_iterator i = addr_list.begin(), i_end = addr_list.end();
         i != i_end;
         ++i)
    {
        if (boost::algorithm::iequals(i->second, args_->context->task->email)) return true;
    }
    return false;
}

template <class ForwardTraversalIterator, class Stream>
bool header_handler<ForwardTraversalIterator, Stream>::parse_loop(const string& suid_md5)
{
    string suid_str = "_" + args_->context->task->suid + "_";
    return (suid_md5 == md5_hex(suid_str));
}

template <class ForwardTraversalIterator, class Stream>
void header_handler<ForwardTraversalIterator, Stream>::parse_subject(const string& source)
{
    std::vector<mulca_mime::EncodedWord> vec = mulca_mime::decode_rfc2047(source);
    info_.subject.clear();
    info_.subject.reserve(source.size());
    for (std::vector<mulca_mime::EncodedWord>::const_iterator i = vec.begin(), i_end = vec.end();
         i != i_end;
         ++i)
    {
        try
        {
            info_.subject += yplatform::convert::iconvert_string("utf-8", i->charset, i->word);
        }
        catch (std::exception& e)
        {
            YRIMAP_ERROR(args_->context) << "parseSubject charset=" << i->charset << " word='"
                                         << i->word << "' error: " << e.what();
        }
        catch (...)
        {
            YRIMAP_ERROR(args_->context) << "parseSubject charset=" << i->charset << " word='"
                                         << i->word << "' error: unknown";
        }
    }
}

template <class ForwardTraversalIterator, class Stream>
bool header_handler<ForwardTraversalIterator, Stream>::headerField(
    const range& st,
    const range& eol)
{
    MimeParser::HeaderField headerField(st.begin(), eol.begin());
    if (!headerField.isValid())
    {
        return true;
    }
    string name(headerField.name());
    boost::to_lower(name);
    if (settings_->remove_headers.find(name) != settings_->remove_headers.end()) return true;
    if (boost::iequals(name, "Date"))
    {
        info_.send_date = parse_date(headerField.value());
    }
    else if (boost::iequals(name, "Received"))
    {
        time_t current_received = parse_received(headerField.value());
        if (current_received)
        {
            info_.last_received_date = std::min(info_.last_received_date, current_received);
            info_.max_received_date = std::max(info_.max_received_date, current_received);
        }
    }
    else if (boost::iequals(name, "From"))
    {
        info_.is_sender = parse_from(headerField.value());
    }
    else if (boost::iequals(name, "X-Yandex-Forward"))
    {
        info_.is_loop |= parse_loop(headerField.value());
    }
    else if (boost::iequals(name, "Subject"))
    {
        parse_subject(headerField.value());
    }
    else if (boost::iequals(name, "Message-ID"))
    {
        info_.message_id = headerField.value();
    }
    else if (boost::iequals(name, "X-YANDEX-RECOVERY-EMAIL"))
    {
        info_.is_recovery_email = true;
    }
    stream_ << st;
    return true;
}

template <typename Iterator, typename Stream>
Iterator mimeparser_parse(
    Stream& stream,
    rpop_args_ptr args,
    Iterator b,
    Iterator e,
    message_info& info,
    const settings_ptr& settings)
{
    typedef header_handler<Iterator, Stream> hhandler_t;
    typedef MimeParser::Parsers::HeaderParser<Iterator, hhandler_t> hparser_t;
    hhandler_t hhandler(b, stream, args, info, settings);
    hparser_t hparser(b, hhandler);
    hparser.push(e);
    hparser.stop();
    info = hhandler.info();
    if (info.last_received_date == std::numeric_limits<time_t>::max()) info.last_received_date = 0;
    return hhandler.header_end();
}

string make_xyandex_hint(
    const string& folder_name,
    int hdr_date,
    int received_date,
    bool mark_as_seen,
    bool skip_loop_prevention,
    const std::string& label_id)
{
    static const int FLAG_SEEN = 2048;
    int flags = 0;
    if (mark_as_seen) flags |= FLAG_SEEN;

    std::stringstream ss;
    if (!folder_name.empty()) ss << "folder_path=" << folder_name << "\n";
    if (hdr_date != 0) ss << "hdr_date=" << hdr_date << "\n";
    if (received_date != 0) ss << "received_date=" << received_date << "\n";
    if (flags != 0) ss << "mixed=" << flags << "\n";
    if (skip_loop_prevention) ss << "skip_loop_prevention=1\n";

    if (mark_as_seen) ss << "notify=0\n";
    else
        ss << "notify=1\n";

    ss << "forward=1\n";
    if (label_id.size())
    {
        ss << "lid=" << label_id << "\n";
    }
    return encode_base64(ss.str());
}

template <typename Stream>
void add_xyandex_hint(
    Stream& stream,
    const rpop_args_ptr& args,
    const message_info& msg_info,
    const string& sent_folder)
{
    std::time_t tmnow = std::time(0);
    bool is_loop_ignore =
        msg_info.is_loop && msg_info.max_received_date < args->context->task->create_date;
    time_t recv_time =
        msg_info.last_received_date ? msg_info.last_received_date : msg_info.send_date;

    bool mark_as_seen =
        (args->context->task->mark_archive_read && recv_time &&
         tmnow - args->context->task->create_date <= 7 * 24 * 60 * 60 &&
         recv_time < args->context->task->create_date);

    bool send_to_sent = msg_info.is_sender && isGmailEmail(args);
    mark_as_seen |= send_to_sent;

    std::string dest_folder = args->context->task->root_folder;
    if (dest_folder.empty() && send_to_sent)
    {
        dest_folder = sent_folder;
    }

    stream << "X-Yandex-Hint: "
           << make_xyandex_hint(
                  dest_folder,
                  0,
                  recv_time,
                  mark_as_seen,
                  is_loop_ignore,
                  args->context->task->label_id)
           << msg_info.crlf;
}

}

void rpopprocessor_impl::add_rpop_headers(
    std::ostream& stream,
    const rpop_args_ptr& args,
    const message_info& msg_info)
{
    string short_name = args->context->task->login.substr(0, args->context->task->login.find("@"));
    string received_date;
    {
        char timestr[256];
        char zonestr[256];
        time_t rawtime;
        time(&rawtime);
        received_date = rfc822date(&rawtime, timestr, sizeof timestr, zonestr, sizeof zonestr);
    }
    stream << "X-yandex-pop-server: " << args->context->task->server << msg_info.crlf
           << "X-yandex-rpop-id: " << args->context->task->popid << msg_info.crlf
           << "X-yandex-rpop-info: " << short_name << "@" << args->context->task->server
           << msg_info.crlf << "X-Yandex-Suid: " << args->context->task->suid << " "
           << args->context->task->bb_info.getEmail() << msg_info.crlf << "Received: "
           << "from " << short_name << "@" << args->context->task->server << " (["
           << args->context->host_ip << "])" << msg_info.crlf << "\tby mail.yandex.ru with POP3 id "
           << args->context->uniq_id() << msg_info.crlf << "\tfor " << args->context->task->suid
           << "@" << args->context->task->popid << "; " << received_date << msg_info.crlf;
}

std::pair<yplatform::zerocopy::segment, std::size_t> rpopprocessor_impl::insert_headers(
    rpop_args_ptr args,
    const string& message,
    message_info& msg_info)
{
    std::size_t sanitize_start = sanitize_message(message);
    msg_info.crlf = "\r\n";
    string::size_type crlf_test = 1;
    if ((crlf_test = message.find("\n", crlf_test)) != string::npos)
    {
        if (message[crlf_test - 1] != '\r') msg_info.crlf = "\n";
    }
    yplatform::zerocopy::streambuf buf;
    {
        std::ostream stream(&buf);
        add_rpop_headers(stream, args, msg_info);
        std::string::const_iterator i_headers = mimeparser_parse(
            stream, args, message.begin() + sanitize_start, message.end(), msg_info, settings_);
        add_xyandex_hint(stream, args, msg_info, settings_->sent_folder_name);
        stream << msg_info.crlf;
        stream << boost::make_iterator_range(i_headers, message.end());
        stream.flush();
    }
    std::size_t msize = buf.size();
    return std::make_pair(buf.detach(buf.end()), msize);
}

} // namespace yrpopper::processor
