#include <butil/digest.h>
#include <boost/format.hpp>
#include <ymod_blackbox/error.h>
#include <yplatform/encoding/url_encode.h>

#include <common/idna.h>
#include <common/karma.h>
#include <api/error.h>
#include <processor/rpop/impl.h>
#include <ymod_popclient/call.h>
#include <ymod_popclient/errors.h>

namespace yrpopper::processor {

string get_uidl_map_hash(const ymod_pop_client::message_list_t& map)
{
    string source;
    for (ymod_pop_client::message_list_t::const_iterator i = map.begin(), i_end = map.end();
         i != i_end;
         ++i)
    {
        if (i->second.id < 0) continue;
        source += i->second.uidl;
    }
    return md5_hex(source);
}

void rpopprocessor_impl::begin_rpop(const rpop_args_ptr& args)
{
    if (args->context->task->bb_info.getLastUpdated() <
        std::time(0) - settings_->blackbox_cache_invalidate)
        request_blackbox(
            args,
            boost::bind(&rpopprocessor_impl::connect_to_pop3_server, get_shared_from_this(), args));
    else
        connect_to_pop3_server(args);
}

void rpopprocessor_impl::request_blackbox(rpop_args_ptr args, const bb_callback_t& cb)
{
    ymod_blackbox::request areq(args->context, args->context->task->suid, 2, "127.0.0.1");
    areq.load_language = true;
    areq.no_password = true;
    yplatform::future::future<ymod_blackbox::response> aresp =
        yplatform::find<ymod_blackbox::auth>("auth")->authenticate(areq);

    aresp.add_callback(boost::bind(
        &rpopprocessor_impl::handle_blackbox, yplatform::shared_from(this), args, aresp, cb));
}

bool is_removed_user(auth_response_t var)
{
    try
    {
        var.get();
    }
    catch (ymod_blackbox::auth_no_mail_error& e)
    {
        return true;
    }
    catch (...)
    {
        return false;
    }
    return false;
}

void rpopprocessor_impl::handle_blackbox(
    rpop_args_ptr args,
    auth_response_t var,
    const bb_callback_t& cb)
{
    ymod_blackbox::response aresp;
    string err_reason;
    if (var.has_exception())
    {
        aresp.success = false;
        err_reason = get_exception_reason(var);
    }
    else
    {
        aresp = var.get();
        err_reason = aresp.err_str;
    }
    if (!aresp.success)
    {
        YRIMAP_ERROR(args->context) << "blackbox.get: status=error (" << err_reason << "), "
                                    << args->context->task->to_log_string();
        fail_finish(args, code::blackbox_error, is_removed_user(var));
        return;
    }
    if (aresp.storage.empty())
    {
        YRIMAP_ERROR(args->context) << "blackbox.get: status=error (nonexistent user) "
                                    << args->context->task->to_log_string();
        fail_finish(args, code::nonexistent_user, aresp.storage.empty());
        return;
    }
    if (!checkKarma(aresp.karma, aresp.karma_status))
    {
        YRIMAP_ERROR(args->context)
            << "blackbox.get: status=error (bad karma) " << args->context->task->to_log_string();
        fail_finish(args, code::bad_karma, true);
        return;
    }
    YRIMAP_LOG(args->context) << "blackbox.update: status=ok, login=" << aresp.email
                              << ", lang=" << aresp.language;
    args->context->task->bb_info.update(
        aresp.uid, aresp.storage, idna_encode_email(aresp.email), aresp.language);
    cb();
}

void rpopprocessor_impl::connect_to_pop3_server(const rpop_args_ptr& args)
{
    ymod_pop_client::future_connect_result fres =
        yplatform::find<ymod_pop_client::call>("pop_client")
            ->connect(
                args->context,
                args->context->task->server,
                args->context->task->port,
                args->context->task->use_ssl);

    fres.add_callback(boost::bind(
        &rpopprocessor_impl::handle_pop_connect, yplatform::shared_from(this), args, fres));
}

void rpopprocessor_impl::handle_pop_connect(
    rpop_args_ptr args,
    ymod_pop_client::future_connect_result res)
{
    if (res.has_exception())
    {
        YRIMAP_ERROR(args->context)
            << "pop.connect: status=error (" << get_exception_reason(res)
            << "), host=" << args->context->task->server << ", port=" << args->context->task->port
            << ", ssl=" << args->context->task->use_ssl;
        try
        {
            res.get();
        }
        catch (ymod_pop_client::ssl_error& e)
        {
            fail_finish(args, code::ssl_error, true);
        }
        catch (...)
        {
            fail_finish(args, code::connect_error, true);
        }
        return;
    }
    args->context->host_ip = res.get().server_ip;
    ymod_pop_client::future_bool_t fres =
        yplatform::find<ymod_pop_client::call>("pop_client")
            ->login(args->context, args->context->task->login, args->context->task->password);

    fres.add_callback(boost::bind(
        &rpopprocessor_impl::handle_pop_login, yplatform::shared_from(this), args, fres));
}

void rpopprocessor_impl::handle_pop_login(rpop_args_ptr args, ymod_pop_client::future_bool_t res)
{
    if (is_finished(args, code::login_error, "pop.login", res)) return;
    ymod_pop_client::future_msg_list_ptr fres = yplatform::find<ymod_pop_client::call>("pop_client")
                                                    ->load_msg_list(args->context, true, false);

    fres.add_callback(boost::bind(
        &rpopprocessor_impl::handle_pop_list, yplatform::shared_from(this), args, fres));
}

void rpopprocessor_impl::handle_pop_list(
    rpop_args_ptr args,
    ymod_pop_client::future_msg_list_ptr res)
{
    if (is_finished(args, code::message_list_error, "pop.getmessages", res)) return;
    rpop_processor_ctx_ptr ctx = get_ctx(args->context);

    ymod_pop_client::message_list_ptr msg_list = res.get();
    ctx->messages.swap(*msg_list);
    args->process_status->message_count = 0;
    for (ymod_pop_client::message_list_t::const_iterator i = ctx->messages.begin(),
                                                         i_end = ctx->messages.end();
         i != i_end;
         ++i)
    {
        if (i->second.id < 0) continue;
        ++args->process_status->message_count;
    }
    if (!args->is_uidl_hash_disabled)
    {
        string uidl_hash = get_uidl_map_hash(ctx->messages);
        if (uidl_hash == args->context->task->uidl_hash || ctx->messages.empty())
        {
            args->process_status->mailbox_sync = true;
            return pop_quit(args);
        }
        else
        {
            YRIMAP_LOG(args->context)
                << "pop.getmessages: status=ok, messages=" << args->process_status->message_count
                << ", uidl_hash_status=nequal (" << args->context->task->uidl_hash << " <> "
                << uidl_hash << ")";
            args->process_status->uidl_hash = uidl_hash;
        }
    }
    request_blackbox(
        args, boost::bind(&rpopprocessor_impl::load_uidl_set, get_shared_from_this(), args));
}

void rpopprocessor_impl::load_uidl_set(const rpop_args_ptr& args)
{
    future_uidl_map fres = args->dbInterface->loadUidls(args->context, args->context->task->popid);

    fres.add_callback(boost::bind(
        &rpopprocessor_impl::handle_load_uidls, yplatform::shared_from(this), args, fres));
}

void rpopprocessor_impl::handle_load_uidls(rpop_args_ptr args, future_uidl_map res)
{
    if (res.has_exception())
    {
        YRIMAP_ERROR(args->context)
            << "cacher.get: status=error (" << get_exception_reason(res) << ")";
        fail_finish(args, code::cacher_service_error);
        return;
    }
    rpop_processor_ctx_ptr ctx = get_ctx(args->context);
    ctx->uidls.swap(*res.get());
    YRIMAP_LOG(args->context) << "cacher.get: status=ok UIDLs=" << ctx->uidls.size();
    process_message(args, ctx->messages.begin());
}

void rpopprocessor_impl::pop_quit(rpop_args_ptr args)
{
    ymod_pop_client::future_bool_t fres =
        yplatform::find<ymod_pop_client::call>("pop_client")->quit(args->context);

    fres.add_callback(boost::bind(
        &rpopprocessor_impl::handle_pop_quit, yplatform::shared_from(this), args, fres));
}

void rpopprocessor_impl::handle_pop_quit(rpop_args_ptr args, ymod_pop_client::future_bool_t res)
{
    if (res.has_exception())
    {
        YRIMAP_ERROR(args->context)
            << "pop.disconnect: status=error (" << get_exception_reason(res) << ")";
    }
    if (need_save_uidls(args))
    {
        save_uidls(args);
    }
    else
    {
        finish(args);
    }
}

bool rpopprocessor_impl::need_save_uidls(rpop_args_ptr args)
{
    return !args->context->task->leave_msgs && args->process_status->uidl_hash.size() &&
        args->process_status->mailbox_sync;
}

void rpopprocessor_impl::save_uidls(rpop_args_ptr args)
{
    rpop_processor_ctx_ptr ctx = get_ctx(args->context);
    future_void_t fres =
        args->dbInterface->saveUidls(args->context, args->context->task->popid, ctx->new_uidls);

    fres.add_callback(boost::bind(
        &rpopprocessor_impl::handle_save_uidls, yplatform::shared_from(this), args, fres));
}

void rpopprocessor_impl::handle_save_uidls(rpop_args_ptr args, future_void_t res)
{
    if (res.has_exception())
    {
        YRIMAP_ERROR(args->context)
            << "cacher.saveuidl: status=error (" << get_exception_reason(res) << ")";
        args->task_error = code::cacher_service_error;
    }
    finish(args);
}

} // namespace yrpopper::processor
