#include <processor/rpop/impl.h>
#include <processor/rfc822date.h>
#include <common/typed_log.h>
#include <common/util.h>
#include <boost/format.hpp>

namespace yrpopper::processor {

template <typename Stream>
void make_notify_mail(const string& generic, const std::map<string, string>& params, Stream& dest)
{
    string::const_iterator start = generic.begin();
    string::const_iterator end = generic.end();
    for (; start != end; ++start)
    {
        switch (*start)
        {
        case '\r':
        case '\n': // ignore CR and LF symbols - use in generic '\r' and '\n' special symbols
            break;
        case '$': // start variable name
        {
            string::const_iterator i = ++start;
            for (; *i != '$' && i != end; ++i)
                ;
            if (i == end) throw std::runtime_error("Invalid variable name in notify template");
            std::map<string, string>::const_iterator var_value = params.find(string(start, i));
            if (var_value == params.end())
                throw std::runtime_error(
                    "Variable " + string(start, i) + " not found in params map");
            dest << var_value->second;
            start = i;
        }
        break;
        case '\\':
            ++start;
            if (*start == 'n')
            {
                dest << '\n';
                break;
            }
            else if (*start == 'r')
            {
                dest << '\r';
                break;
            }
        default:
            dest << *start;
        }
    }
}

void rpopprocessor_impl::process_message(rpop_args_ptr args, message_iter iter)
{
    args->context->update_ctx();
    if (args->context->is_cancelled())
    {
        args->process_status->uidl_hash.clear();
        pop_quit(args);
        return;
    }
    rpop_processor_ctx_ptr ctx = get_ctx(args->context);

    for (; iter != ctx->messages.end(); ++iter)
    {
        if (iter->second.id < 0) continue;
        uidl_map::const_iterator i_uidl = ctx->uidls.find(iter->second.uidl);

        if (i_uidl == ctx->uidls.end()) break;

        if (!i_uidl->second)
        {
            ctx->new_uidls.push_back(std::make_pair(i_uidl->first, i_uidl->second));
        }

        if (isGmailEmail(args))
        {
            process_load_gmail_msg(args, iter);
            return;
        }
    }
    if (iter == ctx->messages.end())
    {
        pop_quit(args);
        return;
    }
    if (args->context->sent_count > settings_->max_sent_count ||
        args->context->sent_size > settings_->max_sent_size)
    {
        YRIMAP_ERROR(args->context)
            << "process_message: session limits exceeded, breaking at"
            << " count=" << args->context->sent_count << " size=" << args->context->sent_size;
        args->process_status->uidl_hash.clear();
        pop_quit(args);
        return;
    }
    if (!++sent_counter_)
    {
        YRIMAP_ERROR(args->context)
            << "process_message: sent count limits exceeded, breaking at"
            << " count=" << args->context->sent_count << " size=" << args->context->sent_size;
        args->process_status->uidl_hash.clear();
        pop_quit(args);
        return;
    }
    request_message(args, iter);
}

void rpopprocessor_impl::report_message_and_process_next(
    rpop_args_ptr args,
    message_iter iter,
    const std::string& status,
    const std::string& error,
    const message_info& msg_info)
{
    report_message(args, status, error, msg_info);
    process_next_message(args, iter);
}

void rpopprocessor_impl::report_message(
    rpop_args_ptr args,
    const std::string& status,
    const std::string& error,
    const message_info& msg_info,
    const std::string& smtp_response)
{
    typed_log::log_store_message(
        args->context,
        status,
        error,
        "",
        "",
        "pop3_id:" + msg_info.uidl,
        "pop3",
        msg_info.message_id,
        smtp_response);
}

void rpopprocessor_impl::process_next_message(rpop_args_ptr args, message_iter iter)
{
    ++iter;
    process_message(args, iter);
}

void rpopprocessor_impl::process_load_gmail_msg(const rpop_args_ptr& args, message_iter iter)
{
    YRIMAP_ERROR(args->context) << "process_load_gmail_msg message: id=" << iter->second.id
                                << ", uidl=" << iter->second.uidl << " already exists";

    ymod_pop_client::future_string_ptr fres =
        yplatform::find<ymod_pop_client::call>("pop_client")
            ->load_msg(args->context, iter->second.id, -1, false);

    fres.add_callback(boost::bind(
        &rpopprocessor_impl::handle_gmail_message, yplatform::shared_from(this), args, fres, iter));
}

void rpopprocessor_impl::delete_message(rpop_args_ptr args, message_iter iter, bool app_uidl)
{
    ymod_pop_client::future_bool_t fres = yplatform::find<ymod_pop_client::call>("pop_client")
                                              ->delete_msg(args->context, iter->second.id);

    fres.add_callback(boost::bind(
        &rpopprocessor_impl::handle_pop_delete,
        yplatform::shared_from(this),
        args,
        fres,
        iter,
        app_uidl));
}

void rpopprocessor_impl::request_message(rpop_args_ptr args, message_iter iter)
{
    ymod_pop_client::future_string_ptr fres =
        yplatform::find<ymod_pop_client::call>("pop_client")
            ->load_msg(args->context, iter->second.id, -1, false);

    fres.add_callback(boost::bind(
        &rpopprocessor_impl::handle_pop_message, yplatform::shared_from(this), args, fres, iter));
}

void rpopprocessor_impl::handle_pop_message(
    rpop_args_ptr args,
    ymod_pop_client::future_string_ptr res,
    message_iter iter)
{
    message_info msg_info(iter->second.uidl);
    if (res.has_exception())
    {
        YRIMAP_ERROR(args->context)
            << "pop.loadmessage: status=error (" << get_exception_reason(res)
            << "), id=" << iter->second.id << ", uidl=" << iter->second.uidl;
        try
        {
            res.get();
        }
        catch (ymod_pop_client::connection_timeout& e)
        {
            args->task_error = code::timeout_error;
            if (need_save_uidls(args))
            {
                return save_uidls(args);
            }
            else
            {
                return finish(args);
            }
        }
        catch (ymod_pop_client::transport_error& e)
        {
            args->task_error = code::transport_error;
            if (need_save_uidls(args))
            {
                return save_uidls(args);
            }
            else
            {
                return finish(args);
            }
        }
        catch (const std::exception& e)
        {
            args->process_status->uidl_hash.clear();
            report_message_and_process_next(
                args, iter, "error", "load message error: "s + e.what(), msg_info);
            return;
        }
    }

    ymod_pop_client::string_ptr source_message = res.get();
    if (source_message->empty())
    {
        YRIMAP_ERROR(args->context)
            << "pop.loadmessage: status=error (empty message), id=" << iter->second.id
            << ", uidl=" << iter->second.uidl;
        args->process_status->uidl_hash.clear();
        report_message_and_process_next(args, iter, "error", "empty message body", msg_info);
        return;
    }

    // send message
    yplatform::zerocopy::segment result_message;
    std::size_t result_message_size = 0;
    try
    {
        std::pair<yplatform::zerocopy::segment, std::size_t> rmessage =
            insert_headers(args, *source_message, msg_info);
        result_message = rmessage.first;
        result_message_size = rmessage.second;
    }
    catch (std::exception& e)
    {
        YRIMAP_ERROR(args->context)
            << "pop.insert_headers: status=error (" << e.what() << "), id=" << iter->second.id
            << ", uidl=" << iter->second.uidl;
        args->process_status->uidl_hash.clear();
        report_message_and_process_next(
            args, iter, "error", "add headers error: "s + e.what(), msg_info);
        return;
    }

    std::size_t source_size = 0;
    bool is_email_big = (result_message_size > settings_->max_mail_size);
    if (is_email_big)
    {
        YRIMAP_ERROR(args->context)
            << "process_message: status=error (message size exceeds the limit, size="
            << result_message_size << ",id=" << iter->second.id << ", uidl=" << iter->second.uidl
            << ")";
        if (settings_->big_mail_notify.empty())
        {
            append_uidl(args, iter, result_message_size);
            return;
        }
        try
        {
            source_size = result_message_size;
            std::map<string, string> params;
            {
                char timestr[256];
                char zonestr[256];
                time_t rawtime;
                time(&rawtime);
                params["DATE"] =
                    rfc822date(&rawtime, timestr, sizeof timestr, zonestr, sizeof zonestr);
            }
            {
                std::ostringstream message_date_str;
                boost::posix_time::ptime message_date;
                yrpopper::to_local_time(msg_info.send_date, message_date);
                message_date_str << message_date.date().day().as_number() << "."
                                 << message_date.date().month().as_number() << "."
                                 << message_date.date().year();
                params["SOURCE_DATE"] = message_date_str.str();
            }
            params["EMAIL"] = args->context->task->bb_info.getEmail();
            params["SUBJECT"] = msg_info.subject;
            params["LOGIN"] = args->context->task->login;
            params["SERVER"] = args->context->task->server;
            yplatform::zerocopy::streambuf notify_buf;
            {
                std::ostream notify_stream(&notify_buf);
                add_rpop_headers(notify_stream, args, msg_info);
                make_notify_mail(
                    settings_->big_mail_notify.get_email(
                        args->context->task->bb_info.getLanguage(), settings_->default_language),
                    params,
                    notify_stream);
                notify_stream << "\r\n.\r\n";
            }
            result_message = notify_buf.detach(notify_buf.end());
        }
        catch (std::exception& e)
        {
            YRIMAP_ERROR(args->context)
                << "create_bigmailnotify: status=error (" << e.what() << ")";
            args->process_status->uidl_hash.clear();
            report_message_and_process_next(
                args, iter, "error", "add headers error: "s + e.what(), msg_info);
            return;
        }
        catch (...)
        {
            YRIMAP_ERROR(args->context) << "create_bigmailnotify: status=error (unknown exception "
                                        << __FILE__ << ":" << __LINE__ << ")";
            args->process_status->uidl_hash.clear();
            report_message_and_process_next(args, iter, "error", "add headers error", msg_info);
            return;
        }
        YRIMAP_LOG(args->context) << "create_bigmailnotify: status=ok";
    }

    if (msg_info.is_recovery_email)
    {
        YRIMAP_ERROR(args->context)
            << "process_message: status=error (recovery message), id=" << iter->second.id
            << ", uidl=" << iter->second.uidl;
        append_uidl(args, iter, result_message_size);
        return;
    }

    auto fres =
        yplatform::find<yrpopper::smtp::module>("send_module")
            ->sendMessage(args->context, args->context->task->bb_info.getEmail(), result_message);
    fres.add_callback(boost::bind(
        &rpopprocessor_impl::handle_smtp_message,
        yplatform::shared_from(this),
        args,
        result_message,
        fres,
        iter,
        msg_info,
        source_size));
}

void rpopprocessor_impl::handle_pop_delete(
    rpop_args_ptr args,
    ymod_pop_client::future_bool_t res,
    message_iter iter,
    bool app_uidl)
{
    if (res.has_exception())
    {
        YRIMAP_ERROR(args->context)
            << "pop.deletemessage: status=error (" << get_exception_reason(res)
            << "), id=" << iter->second.id << ", uidl=" << iter->second.uidl << ")";
    }
    else
    {
        YRIMAP_LOG(args->context) << "pop.deletemessage: status=ok, id=" << iter->second.id
                                  << ", uidl=" << iter->second.uidl;
    }
    if (app_uidl) append_uidl(args, iter);
    else
        process_next_message(args, iter);
}

void rpopprocessor_impl::handle_gmail_message(
    const rpop_args_ptr& args,
    ymod_pop_client::future_string_ptr res,
    message_iter iter)
{
    if (res.has_exception())
    {
        YRIMAP_ERROR(args->context)
            << "pop.gmailmessage: status=error (" << get_exception_reason(res)
            << "), id=" << iter->second.id << ", uidl=" << iter->second.uidl;

        try
        {
            res.get();
        }
        catch (ymod_pop_client::connection_timeout& e)
        {
            args->task_error = code::timeout_error;
            return finish(args);
        }
        catch (ymod_pop_client::transport_error& e)
        {
            args->task_error = code::transport_error;
            return finish(args);
        }
        catch (const std::exception& e)
        {
        }
    }
    args->process_status->uidl_hash.clear();
    process_next_message(args, iter);
    return;
}

} // yrpopper::processor
