#include <processor/rpop/impl.h>
#include <boost/format.hpp>
#include <yplatform/encoding/url_encode.h>

namespace yrpopper::processor {

void rpopprocessor_impl::handle_smtp_message(
    rpop_args_ptr args,
    yplatform::zerocopy::segment message,
    Future<smtp::SMTPResult> res,
    message_iter iter,
    const message_info& info,
    std::size_t source_size)
{
    ++args->context->sent_count;
    args->context->sent_size += message.size();

    if (res.has_exception())
    {
        std::string reason;
        std::string smtpResponse;
        try
        {
            res.get();
        }
        catch (const smtp::SMTPException& e)
        {
            reason = e.what();
            smtpResponse = e.smtpResponse;
        }
        catch (const std::exception& e)
        {
            reason = e.what();
        }
        YRIMAP_ERROR(args->context)
            << "smtp.send: status=error (" << reason << "), id=" << iter->second.id
            << ", uidl=" << iter->second.uidl;
        report_message(args, "error", "smtp save error: " + reason, info, smtpResponse);

        if (save_message(args, message))
        {
            YRIMAP_LOG(args->context)
                << "smtp.save: status=ok, id=" << iter->second.id << ", uidl=" << iter->second.uidl;
        }
        else
        {
            fail_finish(args, code::smtp_save_error, false);
            return;
        }
    }
    else
    {
        auto smtpRes = res.get();
        report_message(args, "success", "", info, smtpRes.response);
    }

    if (args->context->task->leave_msgs || source_size)
    {
        append_uidl(args, iter, source_size);
    }
    else
    {
        delete_message(args, iter);
    }
}

bool rpopprocessor_impl::save_message(rpop_args_ptr args, yplatform::zerocopy::segment message)
{
    if (settings_->save_path.empty()) return false;

    srandom(time(NULL));
    try
    {
        for (unsigned i = 0; i < settings_->save_retries; ++i)
        {
            std::ostringstream ss;

            // Generate path to file
            ss << settings_->save_path;
            ss << "/XXXXXX";

            std::string buff = ss.str();

            // Generate temp name
            int temp_file_fd = 0;
            if ((temp_file_fd = mkstemp(&buff[0])) == -1)
            {
                YRIMAP_ERROR(args->context)
                    << "save_message: status=error (can`t create temp file, description: '"
                    << ::strerror(errno) << "')";
                return false;
            }
            ::close(temp_file_fd);
            ::remove(&buff[0]);
            // Append user email
            ss.str("");
            ss << buff << "_" << args->context->task->bb_info.getEmail();

            // Check if file exists
            struct stat st_file_info;
            if (stat(ss.str().c_str(), &st_file_info) == 0) continue;

            // Write message to file
            std::ofstream ofs;
            ofs.open(ss.str().c_str(), std::ofstream::out);
            if (!ofs.is_open())
            {
                YRIMAP_ERROR(args->context)
                    << "save_message: status=error (can`t open temp file " << ss.str() << ")";
                return false;
            }
            auto i_begin = message.begin_fragment();
            auto i_next = message.begin_fragment();
            ++i_next;
            for (auto i = i_begin, i_end = message.end_fragment(); i != i_end; ++i)
            {
                if (i == i_begin || i_next == i_end)
                {
                    const auto data_start = (i == i_begin ? message.head() : (*i)->buff().first);
                    const auto data_end =
                        (i_next == i_end ? message.tail() :
                                           (*i)->buff().first + (*i)->buff().second);
                    ofs.write(data_start, data_end - data_start);
                }
                else
                {
                    ofs.write((*i)->buff().first, (*i)->buff().second);
                }
                if (i_next != i_end) ++i_next;
            }
            ofs.close();
            return true;
        }
    }
    catch (std::exception& e)
    {
        YRIMAP_ERROR(args->context)
            << "save_message: status=error (std::exception: " << e.what() << ")";
        return false;
    }
    catch (...)
    {
        YRIMAP_ERROR(args->context) << "save_message: status=error (unknown exception)";
        return false;
    }
    YRIMAP_ERROR(args->context) << "save_message: status=error (all retries invalid)";
    return false;
}

void rpopprocessor_impl::append_uidl(
    rpop_args_ptr args,
    message_iter iter,
    std::size_t /* source_size */)
{
    future_void_t fres =
        args->dbInterface->appendUidl(args->context, args->context->task->popid, iter->second.uidl);

    fres.add_callback(boost::bind(
        &rpopprocessor_impl::handle_append_uidl, yplatform::shared_from(this), args, iter, fres));
}

void rpopprocessor_impl::handle_append_uidl(
    rpop_args_ptr args,
    message_iter iter,
    future_void_t res)
{
    if (res.has_exception())
    {
        YRIMAP_ERROR(args->context)
            << "cacher.appenduidl: status=error (" << get_exception_reason(res) << ")";
        args->task_error = code::cacher_service_error;
        pop_quit(args);
        return;
    }
    YRIMAP_LOG(args->context) << "cacher.appenduidl: status=ok, id=" << iter->second.id
                              << ", uidl=" << iter->second.uidl;
    process_next_message(args, iter);
}

} // yrpopper::processor
