#pragma once

#include <boost/shared_ptr.hpp>
#include <boost/thread/locks.hpp>
#include <boost/thread/mutex.hpp>
#include <yplatform/application/profile.h>
#include <boost/noncopyable.hpp>

namespace yrpopper { namespace processor {

class sent_counter : private boost::noncopyable
{
    typedef boost::posix_time::ptime ptime;
    typedef boost::posix_time::time_duration time_duration;
    typedef boost::mutex mutex_t;
    typedef boost::unique_lock<mutex_t> lock_t;

    struct impl
    {
        impl(const string& n)
            : max_count(0)
            , current_count(0)
            , last_dimension(boost::posix_time::microsec_clock::local_time())
            , name("yrpopper_proc_brake_" + n)
        {
        }

        unsigned max_count;
        unsigned current_count;
        ptime last_dimension;
        time_duration dimension_interval;
        string name;
        mutex_t mux;
    };
    typedef boost::shared_ptr<impl> impl_ptr;

public:
    sent_counter(const string& name) : impl_(new impl(name))
    {
    }

    void max_count(unsigned c)
    {
        lock_t lock(impl_->mux);
        impl_->max_count = c;
    }

    void dimension_interval(const time_duration& dimension_interval)
    {
        lock_t lock(impl_->mux);
        impl_->dimension_interval = dimension_interval;
    }

    bool operator++()
    {
        lock_t lock(impl_->mux);
        if (impl_->current_count < impl_->max_count)
        {
            ++impl_->current_count;
            return true;
        }
        ptime now(boost::posix_time::microsec_clock::local_time());
        if (impl_->last_dimension + impl_->dimension_interval < now)
        {
            yplatform::profile_repository::instance()(impl_->name, impl_->current_count);
            impl_->last_dimension = now;
            impl_->current_count = 0;
            return true;
        }
        return false;
    }

private:
    impl_ptr impl_;
};

} // namespace processor
} // namespace yrpopper
