#include <processor/triggers/abook.h>
#include <yplatform/encoding/url_encode.h>
#include <yplatform/find.h>
#include <boost/format.hpp>

namespace yrpopper::processor::triggers {

void abook::init(const settings_ptr& st)
{
    st_ = st;
    abook_sync_hosts.reserve(st->abook_sync_hosts.size());
    for (const string& url : st->abook_sync_hosts)
    {
        abook_sync_hosts.push_back(yplatform::find<yhttp::call>("http_client")->make_rm_info(url));
    }
    reload(st_);
}

bool abook::is_abook_enable(const rpop_args_ptr& args)
{
    if (!st_->enable_abook_sync || args->context->task->email.empty() || !++brake_counter_)
        return false;
    if (st_->abook_sync_start_date != boost::posix_time::not_a_date_time &&
        args->context->task->create_date < st_->abook_sync_start_date)
        return false;
    if (args->context->task->abook_sync_state != 2 && args->context->task->abook_sync_state <= 4)
        return false;
    if (args->context->task->abook_sync_state > (st_->abook_retries_limit + 2)) return false;
    std::time_t now = std::time(0);
    std::time_t penalty = (args->context->task->abook_sync_state - 2) * 60 * 60;
    return (now - args->context->task->abook_sync_date > penalty);
}

future_void_t abook::operator()(const rpop_args_ptr& args)
{
    promise_void_t prom;
    if (!is_abook_enable(args))
    {
        prom.set(VoidResult());
        return prom;
    }

    handlers::abook_sync_ptr get_handler(new handlers::abook_sync(args->context));
    std::ostringstream os;
    os << boost::format(st_->abook_sync_request_body) %
            yplatform::url_encode(args->context->task->email) %
            yplatform::url_encode(args->context->task->password) %
            yplatform::url_encode(args->context->task->bb_info.getUid());
    ymod_http_client::string_ptr req_body(new string(os.str()));

    yplatform::future::future<void> fres =
        yplatform::find<yhttp::call>("http_client")
            ->post_url(args->context, get_handler, get_host(), st_->abook_sync_request, req_body);

    fres.add_callback(boost::bind(&abook::handle_abook_sync, this, args, get_handler, fres, prom));
    return prom;
}

void abook::handle_abook_sync(
    rpop_args_ptr args,
    handlers::abook_sync_ptr handler,
    yplatform::future::future<void> res,
    promise_void_t prom)
{
    if (res.has_exception())
    {
        YRIMAP_ERROR(args->context)
            << "abook_sync: status=error (" << get_exception_reason(res) << ")";
        if (args->process_status->abook_sync_state == 2) args->process_status->abook_sync_state = 5;
        else
            ++args->process_status->abook_sync_state;
    }
    else if (handler->confirm())
    {
        YRIMAP_LOG(args->context) << "abook_sync: status=ok (popid=" << args->context->task->popid
                                  << ", email=" << args->context->task->email
                                  << ", rec_count=" << handler->rec_count() << ")";
        args->process_status->abook_sync_state = 4;
    }
    else
    {
        YRIMAP_ERROR(args->context)
            << "abook_sync: status=error (confirm tag not found in WMI response)";
        if (args->process_status->abook_sync_state == 2) args->process_status->abook_sync_state = 5;
        else
            ++args->process_status->abook_sync_state;
    }
    prom.set(VoidResult());
}

} // namespace yrpopper::processor::triggers
