#pragma once

#include <iterator>
#include <boost/range/iterator_range.hpp>
#include <sstream>

// State machine used to find (^|\n). (the leading dot token).
class ld_parser
{
public:
    ld_parser() : state_(STATE_0)
    {
    }

    template <class Iterator>
    bool parse(Iterator b, Iterator e, Iterator& tok_b, Iterator& tok_e);

    void reset()
    {
        state_ = STATE_0;
    }

private:
    typedef enum
    {
        STATE_0,     // ^
        STATE_START, // *
    } machine_state_t;
    machine_state_t state_;
};

// Finds the leading dot token in [b, e) input range.
/**
 * If the ld token was found:
 * returns true,
 * tok_b: the beginning of the ld token suffix in [b, e) input range
 * tok_e: iterator pointing directly past the last character of the ld token
 *
 * Otherwise:
 * returns false (needs more text)
 * tok_b: the end of the portion of the parsed text that is certain to belong to the message body
 * (ie not includes the prefix of the ld token) tok_e: the end of the text parsed
 *
 * The caller must guarantee that upon the next invocation of this function [tok_b, tok_e) range
 * should still be valid and tok_e = b
 */
template <class Iterator>
bool ld_parser::parse(Iterator b, Iterator e, Iterator& tok_b, Iterator& tok_e)
{
    Iterator p = b;
    Iterator s = b;

    for (; p != e; ++p)
    {
        typename std::iterator_traits<Iterator>::value_type ch = *p;
        switch (state_)
        {
        case STATE_0: // ^
            if (ch == '.')
            {
                tok_e = p + 1;
                tok_b = s;
                state_ = STATE_START; // ^ -> *
                return true;
            }
            else if (ch == '\n')
                s = p + 1; // ^ -> ^
            else
                state_ = STATE_START; // ^ -> *
            break;

        case STATE_START: // *
            if (ch == '\n')
            {
                state_ = STATE_0; // * -> ^
                s = p + 1;
            }
            break;
        }
    }

    tok_b = (state_ == STATE_START ? p : s);
    tok_e = p;
    return false;
}

template <class Iterator, class Stream>
void perform_dot_stuffing(Iterator beg, Iterator end, Stream& s)
{
    ld_parser q;
    Iterator ld_beg = beg, ld_end = end;
    while (beg != end)
    {
        if (q.parse(beg, end, ld_beg, ld_end))
        {
            s << boost::make_iterator_range(beg, ld_beg) << "..";
            beg = ld_end;
        }
        else
        {
            s << boost::make_iterator_range(beg, end);
            return;
        }
    }
}

#ifdef TEST_EOM_PARSER

std::string foo(const std::string& msg)
{
    std::ostringstream os;
    perform_dot_stuffing(msg.begin(), msg.end(), os);
    std::string stuffed_msg = os.str();
    std::cout << "msg: \"" << msg << "\", stuffed msg: \"" << stuffed_msg << "\"" << std::endl;

    return stuffed_msg;
}

int main()
{
    assert(foo(".") == "..");
    assert(foo("..") == "...");
    assert(foo("hel.lo") == "hel.lo");
    assert(foo("hello\r\n.") == "hello\r\n..");
    assert(foo("hello\r\n.\r") == "hello\r\n..\r");
    assert(foo("hello\r\n.\r\n") == "hello\r\n..\r\n");
    assert(foo("hello\r\n..") == "hello\r\n...");
    assert(foo("hello\n.there\n") == "hello\n..there\n");
    assert(foo("hello\r\n.there\r\n") == "hello\r\n..there\r\n");

    return 0;
}

#endif
