#include "processor_header_handler.h"
#include <boost/algorithm/string.hpp>

#include <butil/digest.h>

#include <mimeparser/HeaderField.h>
#include <mimeparser/rfc2822date.h>

namespace yrpopper::processor {

bool ProcessorHeaderHandler::beginHeader(const std::string::const_iterator& /* it */)
{
    return true;
}

bool ProcessorHeaderHandler::endHeader(const std::string::const_iterator& it)
{
    parsed = true;
    lastHeaderEnd = it;
    return true;
}

bool ProcessorHeaderHandler::headerField(const Range& line, const Range& eol)
{
    MimeParser::HeaderField headerField(line.begin(), eol.begin());
    if (!headerField.isValid())
    {
        return true;
    }

    std::string name(headerField.name());
    std::string value(headerField.value());

    if (removeHeaders.count(name)) return true;

    if (boost::iequals(name, "Date"))
    {
        messageInfo->dateHeader = parseDate(value);
    }
    else if (boost::iequals(name, "Received"))
    {
        time_t nextReceived = parseReceived(value);
        if (nextReceived)
        {
            if (!messageInfo->hasReceived)
            {
                messageInfo->hasReceived = true;
                messageInfo->minReceivedDate = nextReceived;
                messageInfo->maxReceivedDate = nextReceived;
            }
            else
            {
                messageInfo->minReceivedDate = std::min(messageInfo->minReceivedDate, nextReceived);
                messageInfo->maxReceivedDate = std::max(messageInfo->maxReceivedDate, nextReceived);
            }
        }
    }
    else if (boost::iequals(name, "X-Yandex-Forward"))
    {
        messageInfo->isLoop |= parseForward(value);
    }
    else if (boost::iequals(name, "X-Yandex-Recovery-Email"))
    {
        messageInfo->isRecovery = true;
    }
    else if (boost::iequals(name, "Message-ID"))
    {
        messageInfo->messageID = value;
    }

    stream << line;
    return true;
}

time_t ProcessorHeaderHandler::parseDate(std::string strDate)
{
    boost::trim(strDate);
    rfc2822::rfc2822date rfcDate(strDate);
    return rfcDate.unixtime();
}

time_t ProcessorHeaderHandler::parseReceived(std::string received)
{
    auto datePos = received.rfind(';');
    if (datePos == received.npos) return 0;

    while (datePos < received.size() && !isalnum(received[datePos]))
        ++datePos;
    if (datePos < received.size()) return parseDate(received.substr(datePos));
    else
        return 0;
}

bool ProcessorHeaderHandler::parseForward(std::string forward)
{
    std::string forwardSource = "_" + suid + "_";
    bool isLoop = (forward == md5_hex(forwardSource));
    return isLoop;
}

MessageInfoPtr ProcessorHeaderHandler::getMessageInfo()
{
    return messageInfo;
}

std::string::const_iterator ProcessorHeaderHandler::getHeadersEnd()
{
    return lastHeaderEnd;
}

} // namespace yrpopper::processor
