#include <processor_ng/processor_service.h>
#include <processor_ng/eom_parser.h>
#include <processor_ng/processor_header_generator.h>
#include <processor/rfc822date.h>
#include <common/typed_log.h>

#include <yplatform/zerocopy/streambuf.h>
#include <yplatform/find.h>
#include <butil/butil.h>
#include <mimeparser/HeaderParser.h>

#include "smtpgate_poster.h"

namespace yrpopper::processor {

void ProcessorService::init(const yplatform::ptree& xml)
{
    this->init_pool();

    L_(debug) << "activating processor::ProcessorService";

    settings->update(xml);
}

Future<ProcessorResult> ProcessorService::processMessage(
    rpop_context_ptr context,
    MessagePtr message)
{
    Promise<ProcessorResult> res;
    auto self = std::dynamic_pointer_cast<ProcessorService>(shared_from_this());
    ioService.post(std::bind(&ProcessorService::doProcessMessage, self, context, message, res));

    return res;
}

void ProcessorService::doProcessMessage(
    rpop_context_ptr context,
    MessagePtr message,
    Promise<ProcessorResult> res)
{
    if (message->start == message->end)
    {
        TASK_LOG(context, error) << "ProcessorService::processMessage: empty or currupted message";
        res.set_exception(ProcessorError("empty or currupted message"));
        return;
    }

    MessageInfoPtr messageInfo;
    yplatform::zerocopy::streambuf buf;
    {
        std::ostream stream(&buf);

        string crlf = getMessageCRLF(message);
        ProcessorHeaderGenerator generator(message, crlf, context, stream);

        generator.generateBaseHeaders();
        auto headerHandler = processMessageHeaders(context, message, stream);

        messageInfo = headerHandler->getMessageInfo();
        if (messageInfo->isRecovery)
        {
            res.set_exception(ProcessorError("message is recovery"));
            return;
        }

        generator.generateXYandexHint(messageInfo);
        stream << crlf; // separate headers from body
        auto headersEnd = headerHandler->getHeadersEnd();
        if (!headerHandler->isParsed())
        {
            headersEnd = message->start;
        }
        processMessageBody(message, headersEnd, stream);

        stream.flush();
    }

    if (buf.size() > settings->maxLetterSize)
    {
        res.set_exception(ProcessorError("message size is too big"));
        return;
    }

    sendMessageSmtp(context, buf.detach(buf.end()), messageInfo, res);
}

string ProcessorService::getMessageCRLF(MessagePtr message)
{
    string res = "\r\n";

    auto pos = std::find(message->start, message->end, '\n');
    if (pos == message->body->end())
    {
        return res;
    }

    if (pos == message->body->begin())
    {
        res = "\n";
    }
    else if (*(pos - 1) != '\r')
    {
        res = "\n";
    }

    return res;
}

ProcessorHeaderHandlerPtr ProcessorService::processMessageHeaders(
    rpop_context_ptr context,
    MessagePtr message,
    std::ostream& stream)
{
    ProcessorHeaderHandlerPtr handler = std::make_shared<ProcessorHeaderHandler>(
        settings->removeHeaders, context->task->suid, stream);

    MimeParser::Parsers::HeaderParser<string::const_iterator, ProcessorHeaderHandler> parser(
        message->start, *handler);
    parser.push(message->end);
    parser.stop();

    return handler;
}

void ProcessorService::processMessageBody(
    MessagePtr message,
    string::const_iterator headersEnd,
    std::ostream& stream)
{
    if (message->dotStuffed)
    {
        stream << boost::iterator_range<string::const_iterator>(headersEnd, message->end);
    }
    else
    {
        perform_dot_stuffing(headersEnd, message->end, stream);
    }
    if (*(message->end - 1) == '\n')
    {
        stream << ".\r\n";
    }
    else
    {
        stream << "\r\n.\r\n";
    }
}

void ProcessorService::sendMessageSmtp(
    rpop_context_ptr context,
    const yplatform::zerocopy::segment& message,
    MessageInfoPtr messageInfo,
    Promise<ProcessorResult> res)
{
    auto smtpClient = yplatform::find<yrpopper::smtp::module>("send_module");
    auto smtpRes = smtpClient->sendMessage(context, context->task->bb_info.getEmail(), message);

    auto self = std::dynamic_pointer_cast<ProcessorService>(shared_from_this());
    smtpRes.add_callback(
        std::bind(&ProcessorService::handleSmtpResult, self, context, messageInfo, res, smtpRes));
}

void ProcessorService::handleSmtpResult(
    rpop_context_ptr context,
    MessageInfoPtr messageInfo,
    Promise<ProcessorResult> res,
    Future<smtp::SMTPResult> smtpResFuture)
{
    if (smtpResFuture.has_exception())
    {
        auto msg = get_exception_reason(smtpResFuture);
        TASK_LOG(context, error) << "SMTP send error: " << msg;
        res.set_exception(ProcessorError(msg));
        return;
    }
    auto smtpRes = smtpResFuture.get();
    ProcessorResult result;
    result.smtpResponse = smtpRes.response;
    result.messageId = messageInfo->messageID;
    res.set(result);
}

} // namespace yrpopper::processor

#include <yplatform/module_registration.h>
DEFINE_SERVICE_OBJECT(yrpopper::processor::ProcessorService)
