#pragma once

#include <common/http_loader.h>

#include <yplatform/zerocopy/segment.h>
#include <yplatform/find.h>

namespace yrpopper { namespace processor {

struct PostResult
{
    bool success = false;

    std::string message;
    std::string sessionId;
};

using PostResultFuture = yplatform::future::future<PostResult>;
using PostResultPromise = yplatform::future::promise<PostResult>;

class SmtpgatePoster : public JsonHttpLoader<HttpPost>
{
public:
    SmtpgatePoster(rpop_context_ptr context, std::string host, size_t retries)
        : JsonHttpLoader(context, host, retries), context(context)
    {
    }

    PostResultFuture postMessage(
        const yplatform::zerocopy::segment& message,
        std::string myHostname)
    {
        auto& bbInfo = context->task->bb_info;
        request = "/collect/" + bbInfo.getUid() + "?email=" + bbInfo.getEmail() +
            "&suid=" + context->task->suid;

        // TODO not supported by ymod_blackbox
        //+ "&country=" + bbInfo.getCountry()
        //+ "&phone=" + bbInfo.getPhoneConfirmed()

        +"&requestId=" + context->uniq_id() + "&host=" + myHostname;

        post = boost::make_shared<std::string>(message.begin(), message.end());

        load();
        return postResultPromise;
    }

    virtual std::string getRequestData()
    {
        return request;
    }
    virtual http_string_ptr getPostData()
    {
        return post;
    }

    virtual void handleError(const std::string& errorMessage)
    {
        auto strData = data.str();
        if (strData.empty())
        {
            PostResult res;
            res.message = errorMessage;
            postResultPromise.set(res);
            return;
        }

        std::string errDescription;
        std::string sessionId;
        try
        {
            boost::property_tree::ptree errorPtree;
            boost::property_tree::read_json(this->data.seekg(std::ios_base::beg), errorPtree);
            errDescription = errorPtree.get<std::string>("message");
            sessionId = errorPtree.get<std::string>("session_id");
        }
        catch (...)
        {
            errDescription = std::move(data.str());
        }

        PostResult res;
        res.sessionId = sessionId;
        res.message = errorMessage + ": " + errDescription;
        postResultPromise.set(res);
    }

protected:
    virtual void handleJson(boost::property_tree::ptree&& responsePtree)
    {
        PostResult res;
        res.success = true;
        res.sessionId = responsePtree.get("session_id", "");
        res.message = responsePtree.get("message", "");

        postResultPromise.set(res);
    }

protected:
    rpop_context_ptr context;
    PostResultPromise postResultPromise;

    std::string request;
    http_string_ptr post;
};

} // namespace processor
} // namespace yrpopper
