#pragma once

#include <scheduler/task_index.h>

#include <unordered_map>

namespace yrpopper { namespace scheduler {

class task_chunk
{
public:
    typedef boost::recursive_mutex mutex_t;
    typedef std::unordered_map<popid_t, task_index> task_index_list;
    typedef boost::unique_lock<mutex_t> lock_t;

private:
    struct impl
    {
        impl(uint64_t i = 0) : id(i)
        {
        }

        uint64_t id;
        task_index_list tasks; // popid: task_index
    };

public:
    task_chunk() : impl_(new impl)
    {
    }

    task_chunk(uint64_t chunk_id) : impl_(new impl(chunk_id))
    {
    }

    uint64_t id() const
    {
        return impl_->id;
    }

    void release()
    {
        for (task_index_list::iterator i = impl_->tasks.begin(), i_end = impl_->tasks.end();
             i != i_end;
             ++i)
        {
            i->second.release();
        }
    }

    void pause()
    {
        for (task_index_list::iterator i = impl_->tasks.begin(), i_end = impl_->tasks.end();
             i != i_end;
             ++i)
        {
            i->second.pause();
        }
    }

    bool is_released()
    {
        for (task_index_list::iterator i = impl_->tasks.begin(), i_end = impl_->tasks.end();
             i != i_end;
             ++i)
        {
            if (!i->second.is_released()) return false;
        }
        return true;
    }

    const task_index_list& tasks() const
    {
        return impl_->tasks;
    }

    task_index_list not_released_tasks(boost::optional<std::size_t> limit) const
    {
        task_index_list res;
        for (auto&& [popid, task] : impl_->tasks)
        {
            if (task.is_released()) continue;

            if (limit && res.size() >= *limit)
            {
                break;
            }

            res.insert({ popid, task });
        }
        return res;
    }

    std::pair<task_index*, bool> insert(const task_ptr& t)
    {
        task_index_list::iterator i = impl_->tasks.find(t->popid);
        if (i == impl_->tasks.end())
        {
            i = impl_->tasks.insert(std::make_pair(t->popid, task_index(t))).first;
            return std::make_pair(&i->second, true);
        }
        return std::make_pair(&i->second, false);
    }

    void erase(const popid_t& id)
    {
        impl_->tasks.erase(id);
    }

    const impl* get_impl() const
    {
        return impl_.get();
    }

private:
    boost::shared_ptr<impl> impl_;
};

}}
