#include <scheduler/database.h>

#include <yplatform/active/callback.h>
#include <yplatform/find.h>
#include <yplatform/future/multi_future.hpp>

#include <boost/format.hpp>

namespace yrpopper::scheduler {

namespace detail {

yplatform::future::future<task_ptr_list> decrypt_task_list(
    const task_ptr_list& task_list,
    const versioned_keys_t& dkeys,
    boost::shared_ptr<yplatform::active::pool> pool)
{
    auto task_decryption_futures =
        std::make_shared<std::list<yplatform::future::future<task_ptr>>>();

    for (auto&& task : task_list)
    {
        auto is_oauth = !task->oauth_refresh_token.empty();
        yplatform::future::promise<task_ptr> prom;
        auto decrypt = [is_oauth, prom, dkeys_ = dkeys, task](auto /*ctx*/) mutable {
            try
            {
                auto& word_to_decrypt = is_oauth ? task->oauth_refresh_token : task->password;
                word_to_decrypt = decrypt_password(word_to_decrypt, dkeys_, task->suid);
                prom.set(task);
            }
            catch (const std::exception& e)
            {
                YLOG_G(error) << "skipping collector popid=" << task->popid
                              << " because of decrypt password exception: " << e.what();
                prom.set(nullptr); // Set nullptr instead of exception to prevent future_multi_and
                                   // result from failing.
            }
        };

        auto enqueued = pool->enqueue_method(decrypt, [prom, task]() mutable {
            YLOG_G(error) << "task decryption cancelled, popid=" << task->popid;
            prom.set(nullptr);
        });
        if (!enqueued)
        {
            YLOG_G(error) << "skipping collector popid=" << task->popid
                          << " because of `active::pool` enqueue failure";
            prom.set(nullptr);
        }

        task_decryption_futures->push_back(prom);
    }

    return yplatform::future::future_multi_and(*task_decryption_futures)
        .then([task_decryption_futures](auto /*wait_future*/) {
            task_ptr_list tasks;
            for (auto task_future : *task_decryption_futures)
            {
                auto task = task_future.get();
                if (task) tasks.push_back(task);
            }
            return tasks;
        });
}

} // namespace detail

void database::start()
{
    handle_scan(boost::system::error_code());
    handle_task_update(boost::system::error_code());
}

void database::stop()
{
    lock_t lock(mux_);
    stopped_ = true;
    scan_timer_->cancel();
    task_update_timer_->cancel();
    for (chunk_dict_t::iterator i = chunks_.begin(), i_end = chunks_.end(); i != i_end; ++i)
        i->second.release();
}

void database::handle_scan(const boost::system::error_code& e)
{
    if (e == boost::asio::error::operation_aborted)
    {
        return;
    }
    else if (e)
    {
        YRIMAP_DISPATCH_ERROR(settings_->scheduler_pq_ctx) << "handle_scan error:" << e.message();
    }
    yplatform::active::make_callback(
        settings_->pool, boost::bind(&database::update_chunks, shared_from_this()))();
}

void database::handle_task_update(const boost::system::error_code& e)
{
    if (e == boost::asio::error::operation_aborted)
    {
        return;
    }
    else if (e)
    {
        YRIMAP_DISPATCH_ERROR(settings_->scheduler_pq_ctx)
            << "handle_task_update error:" << e.message();
    }
    yplatform::active::make_callback(
        settings_->pool, boost::bind(&database::refresh_task_list, shared_from_this()))();
}

void database::update_chunks()
{
    // call pq update method
    future_void_t fres =
        db_interface_->updateChunks(settings_->scheduler_pq_ctx, settings_->my_owner_name);

    fres.add_callback(yplatform::active::make_callback(
        settings_->pool, boost::bind(&database::handle_update_chunks, shared_from_this(), fres)));
}

void database::handle_update_chunks(future_void_t res)
{
    if (res.has_exception())
    {
        YRIMAP_DISPATCH_ERROR(settings_->scheduler_pq_ctx)
            << "got exception in update chunks (" << get_exception_reason(res) << ")";
        next_iteration();
        return;
    }

    future_chunk_requester_dict fres =
        db_interface_->listChunks(settings_->scheduler_pq_ctx, settings_->my_owner_name);

    fres.add_callback(yplatform::active::make_callback(
        settings_->pool,
        boost::bind(&database::handle_update_chunk_list, shared_from_this(), fres)));
}

void database::handle_update_chunk_list(future_chunk_requester_dict res)
{
    if (res.has_exception())
    {
        YRIMAP_DISPATCH_ERROR(settings_->scheduler_pq_ctx)
            << "got exception in update chunk list from (" << get_exception_reason(res) << ")";
        next_iteration();
        return;
    }

    std::ostringstream log_stream;
    bool stream_not_empty = false;

    lock_t lock(mux_);
    chunk_dict_t chunks = chunks_;

    auto chunk_requesters_ptr = res.get();
    for (auto& [chunk_id, requester] : *chunk_requesters_ptr)
    {
        if (requester.empty())
        {
            if (chunks.find(chunk_id) == chunks.end())
            {
                if (chunks_to_release_.find(chunk_id) == chunks_to_release_.end())
                {
                    if (stream_not_empty) log_stream << ", ";
                    else
                        stream_not_empty = true;
                    log_stream << "id=" << chunk_id << ", req='" << requester << "', result=added";
                    chunks_.insert(std::make_pair(chunk_id, task_chunk(chunk_id)));
                    settings_->stat->add_chunk("", chunk_id, 0);
                }
                else
                {
                    if (stream_not_empty) log_stream << ", ";
                    else
                        stream_not_empty = true;
                    log_stream << "id=" << chunk_id << ", req='" << requester
                               << "', result=chunks_to_release";
                }
            }
            else
            {
                chunks.erase(chunk_id);
            }
        }
        else if (chunks.find(chunk_id) == chunks.end())
        {
            if (stream_not_empty) log_stream << ", ";
            else
                stream_not_empty = true;
            log_stream << "id=" << chunk_id << ", req='" << requester
                       << "', result=added_to_releasing";
            chunks_to_release_.insert(std::make_pair(chunk_id, task_chunk(chunk_id)));
        }
        else
        {
            if (stream_not_empty) log_stream << ", ";
            else
                stream_not_empty = true;
            log_stream << "id=" << chunk_id << ", req='" << requester << "', result=released";
        }
    }

    for (chunk_dict_t::iterator i = chunks.begin(), i_end = chunks.end(); i != i_end; ++i)
    {
        if (chunk_requesters_ptr->find(i->first) == chunk_requesters_ptr->end())
        {
            if (stream_not_empty) log_stream << ", ";
            else
                stream_not_empty = true;
            log_stream << "id=" << i->first << ", req='', result=not_found";
        }
        i->second.release();
        chunks_to_release_.insert(std::make_pair(i->first, i->second));
        settings_->stat->release_chunk("", i->first);
        for (task_chunk::task_index_list::const_iterator i_task = i->second.tasks().begin(),
                                                         i_task_end = i->second.tasks().end();
             i_task != i_task_end;
             ++i_task)
        {
            rpoppers_.erase(i_task->second.task()->popid);
            erase_task(i_task->second.task()->popid);
            YRIMAP_LOG(settings_->scheduler_pq_ctx)
                << "task " << i_task->second.task()->popid << " erase from " << i->second.id()
                << ":" << static_cast<const void*>(i->second.get_impl()) << " chunk";
        }
        chunks_.erase(i->first);
    }
    lock.unlock();

    if (stream_not_empty)
    {
        YRIMAP_LOG(settings_->scheduler_pq_ctx) << "update_chunk_list: " << log_stream.str();
    }

    release_chunks();
}

void database::release_chunks()
{
    for (chunk_dict_t::iterator i = chunks_to_release_.begin(), i_end = chunks_to_release_.end();
         i != i_end;)
    {
        if (i->second.is_released())
        {
            YRIMAP_LOG(settings_->scheduler_pq_ctx) << "chunk " << i->first << " released";
            future_void_t fres = db_interface_->releaseChunk(
                settings_->scheduler_pq_ctx, settings_->my_owner_name, i->first);

            fres.add_callback(yplatform::active::make_callback(
                settings_->pool,
                boost::bind(&database::handle_release_chunk, shared_from_this(), fres)));

            i = chunks_to_release_.erase(i);
        }
        else
        {
            if (settings_->not_released_tasks_logging.enabled)
            {
                auto popid = i->first;
                auto tasks_sample = i->second.not_released_tasks(
                    settings_->not_released_tasks_logging.tasks_sample_limit);
                log_not_released_tasks(popid, tasks_sample);
            }
            ++i;
        }
    }
    get_chunk_to_request();
}

void database::log_not_released_tasks(uint64_t chunk_id, const task_chunk::task_index_list& tasks)
{
    std::stringstream stream;
    for (auto&& [popid, task] : tasks)
    {
        stream << "popid=" << std::to_string(popid)
               << ";ctx=" << (task.ctx() ? task.ctx()->uniq_id() : "no_context") << ", ";
    }
    YRIMAP_LOG(settings_->scheduler_pq_ctx) << "waiting for chunk_id=" << chunk_id
                                            << ", remaining collectors: " << stream.str() << "...";
}

void database::handle_release_chunk(future_void_t res)
{
    if (res.has_exception())
    {
        YRIMAP_DISPATCH_ERROR(settings_->scheduler_pq_ctx)
            << "got exception in release chunk (" << get_exception_reason(res)
            << ") it will be choose later";
    }
}

void database::get_chunk_to_request()
{
    future_uint64_t fres =
        db_interface_->getNewChunk(settings_->scheduler_pq_ctx, settings_->my_owner_name);

    fres.add_callback(yplatform::active::make_callback(
        settings_->pool,
        boost::bind(&database::handle_to_request_chunk, shared_from_this(), fres)));
}

void database::handle_to_request_chunk(future_uint64_t res)
{
    if (res.has_exception())
    {
        YRIMAP_DISPATCH_ERROR(settings_->scheduler_pq_ctx)
            << "got exception in find chunk to request (" << get_exception_reason(res) << ")";
    }
    else if (res.has_value())
    {
        YRIMAP_LOG(settings_->scheduler_pq_ctx) << "choose chunk";
    }
    next_iteration();
}

database::future_void_t database::update_task(
    const rpop_context_ptr ctx,
    const task_status_ptr status,
    int new_is_on) const
{
    task_info task;

    task.session_duration = status->session_duration;
    task.last_connect = (status->error == code::bad_task) ? ctx->task->last_connect : ::time(0);
    task.last_msg_count = ctx->sent_count;
    task.error_status = is_public_error(status->error) ? status->error.message() : "ok";
    task.bad_retries = status->bad_retries;
    task.is_on = new_is_on;
    task.abook_sync_state = status->abook_sync_state;
    task.validated = status->validated;
    task.uidl_hash = status->uidl_hash.empty() ? ctx->task->uidl_hash : status->uidl_hash;
    task.popid = ctx->task->popid;

    return db_interface_->updateTask(ctx, task, status->server_response);
}

void database::refresh_task_list()
{
    future_task_ptr_list fres =
        db_interface_->getTaskList(settings_->scheduler_pq_ctx, settings_->my_owner_name);

    fres.add_callback(yplatform::active::make_callback(
        settings_->pool, boost::bind(&database::handle_task_list, shared_from_this(), fres)));
}

void database::handle_task_list(future_task_ptr_list fres)
{
    lock_t lock(mux_);
    if (stopped_) return;
    if (fres.has_exception())
    {
        YRIMAP_DISPATCH_ERROR(settings_->scheduler_pq_ctx)
            << "got exception in handle task list (" << get_exception_reason(fres) << ")";
        for (chunk_dict_t::iterator i = chunks_.begin(), i_end = chunks_.end(); i != i_end; ++i)
        {
            i->second.pause();
        }
    }
    else
    {
        auto task_list_ptr = fres.get();
        detail::decrypt_task_list(*task_list_ptr, settings_->dkeys, settings_->pool)
            .then([this, self = shared_from_this()](auto future) {
                lock_t lock(mux_);
                if (future.has_exception())
                {
                    YRIMAP_DISPATCH_ERROR(settings_->scheduler_pq_ctx)
                        << "got exception in decrypt task list (" << get_exception_reason(future)
                        << ")";
                    for (auto&& [id, chunk] : chunks_)
                    {
                        chunk.pause();
                    }
                    return;
                }
                auto tasks = future.get();
                load_tasks(tasks, false);
            });
        YRIMAP_LOG(settings_->scheduler_pq_ctx) << "got " << task_list_ptr->size() << " tasks ";
    }
    task_update_timer_->expires_from_now(settings_->task_update_interval);
    task_update_timer_->async_wait(
        boost::bind(&database::handle_task_update, shared_from_this(), _1));
}

void database::load_tasks(const task_ptr_list& lst, bool add_only)
{
    rpop_set_t new_rpoppers_list;
    for (task_ptr_list::const_iterator i = lst.begin(), i_end = lst.end(); i != i_end; ++i)
    {
        chunk_dict_t::iterator i_chunk = chunks_.find((*i)->chunk_id);
        if (i_chunk == chunks_.end()) continue;

        if (add_only)
        {
            rpoppers_.insert(std::make_pair((*i)->popid, (*i)->chunk_id));
        }
        else
        {
            new_rpoppers_list.insert(std::make_pair((*i)->popid, (*i)->chunk_id));
            rpoppers_.erase((*i)->popid);
        }
        std::pair<task_index*, bool> index = i_chunk->second.insert(*i);
        if (index.second)
        {
            YRIMAP_LOG(settings_->scheduler_pq_ctx)
                << "task " << (*i)->popid << " added to " << i_chunk->second.id() << ":"
                << static_cast<const void*>(i_chunk->second.get_impl()) << " chunk";
            assert(index.first->task()->popid == (*i)->popid);
            insert_task(*index.first);
        }
        else
        {
            index.first->refresh(*i);
        }
    }
    if (add_only) return;
    for (rpop_set_t::const_iterator i = rpoppers_.begin(), i_end = rpoppers_.end(); i != i_end; ++i)
    {
        chunk_dict_t::iterator i_chunk = chunks_.find(i->second);
        assert(i_chunk != chunks_.end());
        YRIMAP_LOG(settings_->scheduler_pq_ctx)
            << "task " << i->first << " erase from " << i_chunk->second.id() << ":"
            << static_cast<const void*>(i_chunk->second.get_impl()) << " chunk";
        i_chunk->second.erase(i->first);
        erase_task(i->first);
    }
    rpoppers_.swap(new_rpoppers_list);
}

void database::next_iteration()
{
    lock_t lock(mux_);
    if (stopped_) return;
    scan_timer_->expires_from_now(settings_->scan_interval);
    scan_timer_->async_wait(boost::bind(&database::handle_scan, shared_from_this(), _1));
}

} // namespace yrpopper::scheduler
