#pragma once

#include <scheduler/chunk.h>
#include <scheduler/settings.h>
#include <db/scheduler_interface.h>
#include <db/interface_provider.h>
#include <common/util.h>
#include <common/task.h>

#include <ymod_pq/call.h>

#include <boost/regex.hpp>

#include <functional>
#include <unordered_map>
#include <memory>

namespace yrpopper { namespace scheduler {

class database : public std::enable_shared_from_this<database>
{
    typedef boost::recursive_mutex mutex_t;
    typedef boost::unique_lock<mutex_t> lock_t;
    typedef std::unordered_map<uint64_t, task_chunk> chunk_dict_t;
    typedef std::unordered_map<popid_t, uint64_t> rpop_set_t;

public:
    typedef std::function<void(const task_index&)> create_callback;
    typedef std::function<void(popid_t)> erase_callback;
    typedef yplatform::future::future<VoidResult> future_void_t;
    typedef yplatform::future::future<bool> future_bool_t;
    typedef yplatform::future::promise<VoidResult> promise_void_t;

    explicit database(settings_ptr st)
        : settings_(st)
        , db_interface_(db::InterfaceProvider::getSchedulerInterface())
        , scan_timer_(new boost::asio::deadline_timer(st->pool->timer_service()))
        , task_update_timer_(new boost::asio::deadline_timer(st->pool->timer_service()))
        , stopped_(false)
    {
    }

    void start();
    void stop();
    void set_task_callbacks(const create_callback& create_cb, const erase_callback& erase_cb)
    {
        create_cb_ = create_cb;
        erase_cb_ = erase_cb;
    }

    future_void_t update_task(
        const rpop_context_ptr ctx,
        const task_status_ptr status,
        int new_is_on) const;

private:
    void handle_scan(const boost::system::error_code& e);
    void handle_task_update(const boost::system::error_code& e);
    void update_chunks();
    void release_chunks();
    void log_not_released_tasks(uint64_t chunk_id, const task_chunk::task_index_list& tasks);
    void get_chunk_to_request();
    void refresh_task_list();
    void handle_update_chunks(future_void_t res);
    void handle_update_chunk_list(future_chunk_requester_dict res);
    void handle_release_chunk(future_void_t res);
    void handle_to_request_chunk(future_uint64_t res);
    void handle_task_list(future_task_ptr_list fres);
    void load_tasks(const task_ptr_list& lst, bool add_only);
    void next_iteration();
    bool stopped()
    {
        lock_t lock(mux_);
        return stopped_;
    }

    void insert_task(const task_index& ti)
    {
        create_cb_(ti);
    }

    void erase_task(popid_t task)
    {
        erase_cb_(task);
    }

    settings_ptr settings_;
    db::SchedulerInterfacePtr db_interface_;
    chunk_dict_t chunks_;
    chunk_dict_t chunks_to_release_;
    rpop_set_t rpoppers_;
    create_callback create_cb_;
    erase_callback erase_cb_;
    mutex_t mux_;
    boost::shared_ptr<boost::asio::deadline_timer> scan_timer_;
    boost::shared_ptr<boost::asio::deadline_timer> task_update_timer_;
    bool stopped_;
};

typedef boost::shared_ptr<database> db_ptr;

}}
