#pragma once

#include <scheduler/exceptions.h>
#include <ymod_pq/bind_array.h>
#include <yplatform/active/callback.h>
#include <yplatform/context_repository.h>
#include <scheduler/limit.h>
#include <scheduler/settings.h>
#include <scheduler/plan_queue.h>
#include <scheduler/scheduler.h>
#include <scheduler/database.h>
#include <processor/processor.h>
#include <ymod_httpclient/call.h>
#include <api/error.h>
#include <api/api.h>
#include <common/typed_log.h>
#include <collector_ng/collector_service.h>
#include <collector_ng/collector_error.h>
#include <db/interface_provider.h>

namespace yrpopper { namespace scheduler {

template <typename Db>
class dispatcher
{
public:
    dispatcher(std::shared_ptr<Db> db)
        : db_(db)
        , sessions_logger(YGLOBAL_LOG_SERVICE, "sessions")
    {
        db_->set_task_callbacks(
            boost::bind(&dispatcher<Db>::push_task, this, _1),
            boost::bind(&dispatcher<Db>::pop_task, this, _1));
        db_->start();
    }

    void init(const settings_ptr& st)
    {
        settings_ = st;
        limits_.set_settings(st);

        TaskRunParamsPtr params = std::make_shared<TaskRunParams>();
        queue_.init(
            boost::bind(&dispatcher<Db>::handle_task, this, _1, "", params, PromiseVoidResultPtr()),
            settings_);
    }

    void start()
    {
        queue_.start();
    }

    void stop()
    {
        queue_.stop();
        db_->stop();
    }

    void reload(const settings_ptr& st)
    {
        settings_ = st;
        limits_.set_settings(st);
    }

    future_void_t manual_run(
        const yplatform::task_context_ptr& ctx,
        const popid_t& id,
        TaskRunParamsPtr params)
    {
        promise_void_t prom;
        std::optional<task_index> ti = queue_.find(id);
        if (ti.has_value()) do_manual_run(ctx, ti.value(), params, prom);
        else
            prom.set_exception(task_not_exists_error());
        return prom;
    }

    void push_task(const task_index& ti)
    {
        queue_.push(ti);
    }

    void pop_task(popid_t task)
    {
        queue_.pop(task);
    }

private:
    void do_manual_run(
        const yplatform::task_context_ptr& ctx,
        const task_index& ti,
        TaskRunParamsPtr params,
        promise_void_t prom)
    {
        PromiseVoidResultPtr promise_ptr;
        if (params->hasFlag(TaskRunParams::RunFlag::Syncronized))
        {
            promise_ptr = std::make_shared<promise_void_t>(prom);
        }

        start_result res = handle_task(ti, ctx->uniq_id(), params, promise_ptr);
        if (!params->hasFlag(TaskRunParams::RunFlag::Syncronized) || res != start_result_ok)
        {
            switch (res)
            {
            case start_result_ok:
            case start_result_err_already:
                prom.set(VoidResult());
                break;
            case start_result_err_penalty:
                prom.set_exception(frequent_run_error());
                break;
            case start_result_err_first:
                prom.set_exception(first_run_interval_error());
                break;
            case start_result_err_host_limit:
                prom.set_exception(host_limits_reached_error());
                break;
            case start_result_err_all_limit:
                prom.set_exception(task_limits_reached_error());
                break;
            default:
                break;
            }
        }
    }

    start_result check_task(const task_ptr& task, bool is_manual) const
    {
        std::time_t penalty{ 0 };
        if (is_manual)
        {
            penalty = calculate_penalty_for_manual_run(task);
        }
        else
        {
            if (!is_first_run_interval_passed(task))
            {
                return start_result_err_first;
            }
            penalty = calculate_penalty(task);
        }

        if (!is_interval_since_last_update_passed(task, penalty))
        {
            return start_result_err_penalty;
        }
        return start_result_ok;
    }

    bool is_first_run_interval_passed(const task_ptr& task) const
    {
        std::time_t now = std::time(nullptr);
        return (now - task->create_date >= settings_->first_run_interval);
    }

    bool is_interval_since_last_update_passed(const task_ptr& task, const std::time_t& penalty)
        const
    {

        std::time_t time_since_last_connect =
            std::time(nullptr) - std::max(task->last_connect, task->touch_time);
        return (time_since_last_connect >= penalty);
    }

    std::time_t calculate_penalty(const task_ptr& task) const
    {
        std::time_t penalty = limits_.get_task_penalty(task->server);
        if (task->bad_retries > settings_->retries_limit)
        {
            penalty = std::max(
                static_cast<std::time_t>(task->bad_retries - settings_->retries_limit) *
                    settings_->penalty,
                penalty);
            penalty = std::min(penalty, static_cast<std::time_t>(60 * 60 * 24));
        }
        else if (task->last_msg_count == 0 && !task->error)
        {
            penalty *= settings_->finished_penalty_multiplier;
        }
        return penalty;
    }

    std::time_t calculate_penalty_for_manual_run(const task_ptr& task) const
    {
        std::time_t penalty = limits_.get_task_penalty(task->server);
        return std::max(settings_->force_penalty, penalty);
    }

    start_result handle_task(
        task_index ti,
        const string& uniq_id,
        TaskRunParamsPtr params,
        PromiseVoidResultPtr prom_ptr)
    {
        start_result res = start_result_ok;
        task_index::lock_t lock(ti.mux());
        rpop_context_ptr ctx(new rpop_context(
            ti.task(),
            uniq_id,
            params->hasFlag(TaskRunParams::RunFlag::Verbose) || settings_->log_extra,
            params->hasFlag(TaskRunParams::RunFlag::Clean),
            prom_ptr));
        ctx->logger() = sessions_logger;
        ctx->logger().set_log_prefix(ctx->uniq_id());

        res = check_task(ti.task(), !uniq_id.empty());
        if (res != start_result_ok) return res;

        res = limits_.run_task(ti.task()->server);
        if (res != start_result_ok)
        {
            YRIMAP_DISPATCH_LOG(ctx)
                << "task skipped (limits exceeded): " << ti.task()->to_log_string();
            return res;
        }

        if (!ti.start(ctx))
        {
            YRIMAP_DISPATCH_LOG(ctx)
                << "task skipped (already started): " << ti.task()->to_log_string();
            limits_.finish_task(ti.task()->server);
            return start_result_err_already;
        }

        lock.unlock();

        yplatform::context_repository::instance().add_context(ctx);

        yrpopper::api::future_task_validity_t fres =
            yplatform::find<yrpopper::api::api>("rpop_api")
                ->check_task_validity(ctx, ti.task()->dbname, ti.task()->suid, *ti.task());

        fres.add_callback(boost::bind(&dispatcher<Db>::handle_task_cont, this, ctx, ti, fres));
        return res;
    }

    void log_task_finish(
        const rpop_context_ptr& ctx,
        const task_status_ptr& status,
        const int _new_is_on)
    {
        std::string uidl_hash_status =
            (status->uidl_hash.size() > 0 && ctx->sent_count == 0) ? "missing" : "ok";

        YRIMAP_DISPATCH_LOG(ctx) << "task finished: status: '" << status->error.message()
                                 << "', is_on:" << _new_is_on << ", uidl_hash: " << uidl_hash_status
                                 << ", penalty:" << calculate_penalty(ctx->task);

        typed_log::log_iteration(ctx, status->error, status->session_duration, status->bad_retries);
    }

    void log_task_finish_with_exception(
        const rpop_context_ptr& ctx,
        const std::string& reason,
        std::time_t duration)
    {
        YRIMAP_ERROR(ctx) << "processor response exception: " << reason;

        error ec = { code::internal_error, reason };
        typed_log::log_iteration(ctx, ec, duration, ctx->task->bad_retries);
    }

    void handle_task_cont(rpop_context_ptr ctx, task_index ti, api::future_task_validity_t result)
    {
        if (result.has_exception())
        {
            try
            {
                result.get();
            }
            catch (std::exception& e)
            {
                YRIMAP_ERROR(ctx) << "[bad task] check_task_validity exception: \"" << e.what()
                                  << "\", " << ti.task()->to_log_string();
            }
            catch (...)
            {
                YRIMAP_ERROR(ctx) << "[bad task] check_task_validity unknown exception, "
                                  << ti.task()->to_log_string();
            }

            processor::promise_task_status_ptr prom;
            task_status stat(code::bad_task, time(0), 0, 0, 0);
            prom.set(task_status_ptr(new task_status(stat)));
            handle_task_finish(ti, ctx, prom);
            return;
        }

        if (result.get() != yrpopper::api::task_pq_ok)
        {
            yrpopper::api::task_validity tv = result.get();
            if (tv == yrpopper::api::task_bad)
            {
                YRIMAP_ERROR(ctx) << "[bad task] disabling: " << ti.task()->to_log_string();

                // disable bad task
                yplatform::find<yrpopper::api::api>("rpop_api")
                    ->enable(ctx, ti.task()->dbname, ti.task()->suid, ti.task()->popid, false);
            }
            else if (tv == yrpopper::api::task_temporary_error)
            {
                YRIMAP_ERROR(ctx) << "temporary task error: " << result.get() << ", "
                                  << ti.task()->to_log_string();
            }
            processor::promise_task_status_ptr prom;
            task_status stat(code::bad_task, time(0), 0, 0, 0);
            prom.set(task_status_ptr(new task_status(stat)));
            handle_task_finish(ti, ctx, prom);
            return;
        }

        if (ti.task()->use_imap)
        {
            startIMAP(ctx, ti);
        }
        else
        {
            startPOP3(ctx, ti);
        }

        YRIMAP_DISPATCH_LOG(ctx) << "task started: " << ti.task()->to_log_string();
    }

    void handle_task_finish(
        task_index ti,
        rpop_context_ptr ctx,
        processor::future_task_status_ptr res)
    {
        settings_->stat->on_task_finished(ctx->task->popid);

        limits_.finish_task(ti.task()->server);

        ti.finish();

        queue_.on_task_finished(ti.task()->popid);

        yplatform::context_repository::instance().rem_context(ctx);

        if (res.has_exception())
        {
            auto reason = get_exception_reason(res);
            log_task_finish_with_exception(ctx, reason, 0);
        }
        else
        {
            task_status_ptr status = res.get();
            int newIsOn = status->bad_retries > settings_->retries_limit ?
                (status->bad_retries > settings_->fatal_retries_limit ? 3 : 2) :
                1;

            log_task_finish(ctx, status, newIsOn);

            ti.task()->touch_time = std::time(0);
            if (status->error != code::bad_task)
            {
                ti.task()->last_connect = std::time(0);
            }

            updateTask(ti, ctx, status, newIsOn);
        }

        if (ctx->manual_run_promise_ptr)
        {
            ctx->manual_run_promise_ptr->set(VoidResult());
        }
    }

    void startPOP3(rpop_context_ptr ctx, task_index ti)
    {
        processor::future_task_status_ptr fres =
            yplatform::find<yrpopper::processor::processor>("rpop_processor")->process(ctx);

        auto taskCallback = yplatform::active::make_callback(
            settings_->pool, boost::bind(&dispatcher<Db>::handle_task_finish, this, ti, ctx, fres));

        fres.add_callback(taskCallback);
    }

    void startIMAP(rpop_context_ptr ctx, task_index ti)
    {
        auto collectorService =
            yplatform::find<yrpopper::collector::CollectorService>("collector_service");
        auto collector = collectorService->getCollector(ctx);
        auto stepRes = collector->step();
        stepRes.add_callback(yplatform::active::make_callback(
            settings_->pool,
            [this, ti, ctx, stepRes](yplatform::active::object_context* /* ignored */) {
                processor::promise_task_status_ptr prom;
                try
                {
                    auto res = stepRes.get();
                    // status, start_time, duration, message_count, bad_retries
                    auto taskStatusPtr = boost::make_shared<task_status>(
                        res->error,
                        res->startTime,
                        std::time(0) - res->startTime,
                        ctx->sent_count,
                        res->badRetries);
                    taskStatusPtr->validated = res->validated;
                    prom.set(taskStatusPtr);
                }
                catch (const std::exception& e)
                {
                    prom.set_exception(e);
                }
                this->handle_task_finish(ti, ctx, prom);
            }));
    }

    void updateTask(task_index& ti, rpop_context_ptr ctx, task_status_ptr status, int new_is_on)
    {
        task_index::lock_t lock(ti.mux());
        auto update_res = db_->update_task(ctx, status, new_is_on);
        lock.unlock();
        ti.refresh(status, new_is_on);
        auto updateCallback = [this, ctx, update_res](yplatform::active::object_context*) {
            if (update_res.has_exception())
            {
                YRIMAP_ERROR(ctx) << "update status in db error: "
                                  << get_exception_reason(update_res);
            }
        };
        update_res.add_callback(yplatform::active::make_callback(settings_->pool, updateCallback));
    }

private:
    std::shared_ptr<Db> db_;
    yplatform::log::source sessions_logger;
    settings_ptr settings_;
    plan_queue queue_;
    limit_controller limits_;
};

}}
