#include <scheduler/impl.h>
#include <common/util.h>
#include <common/host_info.h>
#include <boost/format.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/classification.hpp>
#include <yplatform/active/callback.h>

namespace yrpopper::scheduler {

scheduler_pq_impl::scheduler_pq_impl() : settings_(new settings(get_my_hostname()))
{
    L_(info) << "rpop_scheduler_pq_impl task instaniated";
}

scheduler_pq_impl::~scheduler_pq_impl()
{
    L_(info) << "rpop_scheduler_pq_impl task destroyed";
}

void scheduler_pq_impl::init(const yplatform::ptree& xml)
{
    unsigned nthreads = xml.get("threads", 10);
    unsigned queue_capacity = xml.get("queue_capacity", 10000);
    int enqueue_timeout = xml.get("enqueue_timeout", -1);
    if (enqueue_timeout >= 0)
    {
        settings_->enqueue_timeout = boost::posix_time::milliseconds(enqueue_timeout);
    }
    settings_->queue_process_interval =
        xml.get("queue_process_interval", settings_->queue_process_interval);
    boost::optional<const yplatform::ptree&> custom_owner_name =
        xml.get_child_optional("owner_name");
    if (custom_owner_name)
    {
        settings_->my_owner_name = custom_owner_name->get_value("");
    }
    load_limits(xml);
    settings_->default_limit = xml.get("default_server_max_tasks", settings_->default_limit);
    boost::optional<const yplatform::ptree&> custom_limits =
        xml.get_child_optional("custom_limits");
    if (custom_limits)
    {
        for (yplatform::ptree::const_iterator i = custom_limits->begin(),
                                              i_end = custom_limits->end();
             i != i_end;
             ++i)
        {
            auto host = i->second.get("<xmlattr>.host", "");
            auto max_tasks = i->second.get("<xmlattr>.max_active_tasks", settings_->default_limit);
            auto penalty = i->second.get("<xmlattr>.min_penalty", settings_->penalty);
            settings_->host_limits.insert(
                std::make_pair(host, host_limits_t{ max_tasks, penalty }));
        }
    }
    settings_->max_tasks = xml.get("max_active_tasks", settings_->max_tasks);
    settings_->run_prefix = xml.get("run_prefix", ":2048/api");
    settings_->log_extra = xml.get("log_extra", false);
    settings_->pool.reset(new yplatform::active::pool(
        queue_capacity, "rpop_scheduler_pq", settings_->enqueue_timeout));
    settings_->pool->open(nthreads);

    std::string versioned_keys_file = xml.get("versioned_keys_file", "/etc/yrpop/versioned_keys");
    settings_->dkeys = compute_derived_keys(versioned_keys_file);
    settings_->list_request_timeout =
        xml.get("list_request_timeout", settings_->list_request_timeout);
    settings_->not_released_tasks_logging.enabled = xml.get(
        "not_released_tasks_logging.enabled", settings_->not_released_tasks_logging.enabled);
    settings_->not_released_tasks_logging.tasks_sample_limit =
        xml.get_optional<std::size_t>("not_released_tasks_logging.tasks_sample_limit");

    disp_ = std::make_unique<dispatcher<database>>(std::make_shared<database>(settings_));
    disp_->init(settings_);
}

void scheduler_pq_impl::start()
{
    disp_->start();
}

void scheduler_pq_impl::reload(const yplatform::ptree& xml)
{
    load_limits(xml);
    disp_->reload(settings_);
}

void scheduler_pq_impl::stop()
{
    disp_->stop();
}

void scheduler_pq_impl::fini(void)
{
    settings_->pool->close();
    settings_->pool->clear();
}

void scheduler_pq_impl::load_limits(const yplatform::ptree& xml)
{
    settings_->retries_limit = xml.get("retries_limit", settings_->retries_limit);
    settings_->fatal_retries_limit = xml.get("fatal_retries_limit", settings_->fatal_retries_limit);
    settings_->penalty = xml.get("task_run_interval", settings_->penalty);
    settings_->force_penalty = xml.get("force_run_interval", settings_->force_penalty);
    settings_->finished_penalty_multiplier =
        xml.get("finished_penalty_multiplier", settings_->finished_penalty_multiplier);
    settings_->first_run_interval = xml.get("first_run_interval", settings_->first_run_interval);
    settings_->scan_interval = boost::posix_time::seconds(xml.get("scan_interval", 10));
    settings_->task_update_interval =
        boost::posix_time::seconds(xml.get("task_update_interval", 60));
}

future_void_t scheduler_pq_impl::forceTaskRun(
    const yplatform::task_context_ptr& context,
    const popid_t& id,
    TaskRunParamsPtr params)
{
    return disp_->manual_run(context, id, params);
}

} // namespace yrpopper::scheduler

#include <yplatform/module_registration.h>

DEFINE_SERVICE_OBJECT(yrpopper::scheduler::scheduler_pq_impl)
