#include <scheduler/limit.h>

namespace yrpopper::scheduler {

std::time_t limit_controller::get_task_penalty(const string& host) const
{
    lock_t lock(mux_);
    const string& rhost = real_host(host);
    auto limit = get_limit(rhost);

    return limit.min_penalty;
}

start_result limit_controller::run_task(const string& host)
{
    lock_t lock(mux_);
    if (active_count_ >= settings_->max_tasks) return start_result_err_all_limit;
    const string& rhost = real_host(host);
    uint64_t limit = get_limit(rhost).max_active_tasks;
    if (!limit) return start_result_err_host_limit;
    active_map::iterator i = current_active_.find(rhost);
    if (i == current_active_.end())
    {
        i = current_active_.insert(std::make_pair(rhost, 0)).first;
    }
    if (i->second >= limit) return start_result_err_host_limit;
    ++i->second;
    ++active_count_;
    return start_result_ok;
}

void limit_controller::finish_task(const string& host)
{
    lock_t lock(mux_);
    --active_count_;
    const string& rhost = real_host(host);
    active_map::iterator i = current_active_.find(rhost);
    assert(i != current_active_.end());
    if (--i->second) return;
    current_active_.erase(i);
}

host_limits_t limit_controller::get_limit(const string& host) const
{
    host_limit_map::const_iterator i_limit = settings_->host_limits.find(host);
    if (i_limit == settings_->host_limits.end())
        return { settings_->default_limit, settings_->penalty };
    return i_limit->second;
}

const string& limit_controller::real_host(const string& host) const
{
    host_alias_map::const_iterator i_host = settings_->aliases.find(host);
    if (i_host == settings_->aliases.end()) return host;
    return i_host->second;
}

} // namespace yrpopper::scheduler
