#include <scheduler/plan_queue.h>
#include <yplatform/active/callback.h>
#include <yplatform/application/profile.h>
#include <yplatform/find.h>

#include <collector_ng/collector_service.h>

namespace yrpopper::scheduler {

std::string make_rpop_identifier(const popid_t& id)
{
    return boost::lexical_cast<std::string>(id);
}

void plan_queue::init(const run_handler& h, const settings_ptr& st)
{
    settings_ = st;
    timer_ = std::make_shared<yplatform::time_traits::timer>(settings_->pool->timer_service());
    handler_ = h;
    queue_process_interval_ = settings_->queue_process_interval;
}

void plan_queue::push(const task_index& ti)
{
    popid_t rpop_id = ti.task()->popid;
    lock_t lock(mux_);
    if (map_.find(rpop_id) != map_.end())
    {
        // XXX move to dispatcher
        L_(error) << "error, already in queue: plan_queue::push id=" << rpop_id;
        return;
    }
    map_.emplace(rpop_id, ti);
    pending_tasks_.push_back(rpop_id);
    lock.unlock();
    // XXX move to dispatcher
    L_(info) << "plan_queue::push " << ti.task()->to_log_string();
}

void plan_queue::pop(const popid_t& rpop_id)
{
    lock_t lock(mux_);
    auto it = map_.find(rpop_id);
    if (it == map_.end())
    {
        // XXX move to dispatcher
        L_(error) << "error, no such task in map: plan_queue::pop id=" << rpop_id;
        return;
    }
    auto&& task = it->second;
    // XXX move to dispatcher
    if (task.task()->use_imap)
    {
        auto collectorService =
            yplatform::find<yrpopper::collector::CollectorService>("collector_service");
        collectorService->killCollector(rpop_id);
    }
    map_.erase(it);
    lock.unlock();
    // XXX move to dispatcher
    L_(info) << "plan_queue::pop id=" << rpop_id;
}

std::optional<task_index> plan_queue::find(const popid_t rpop_id) const
{
    lock_t lock(mux_);
    auto it = map_.find(rpop_id);
    if (it == map_.end()) return std::nullopt;
    else
        return it->second;
}

void plan_queue::start()
{
    start_timer();
}

void plan_queue::stop()
{
    lock_t lock(mux_);
    timer_->cancel();
    timer_.reset();
}

void plan_queue::start_timer()
{
    lock_t lock(mux_);
    if (!timer_) return;
    timer_->expires_from_now(queue_process_interval_);
    timer_->async_wait(boost::bind(&plan_queue::handle_timer, this, _1));
}

void plan_queue::on_task_finished(const popid_t& id)
{
    lock_t lock(mux_);
    if (map_.find(id) == map_.end()) return;
    pending_tasks_.push_back(id);
}

void plan_queue::handle_timer(const boost::system::error_code& ec)
{
    if (ec == boost::asio::error::operation_aborted) return;
    yplatform::active::make_callback(settings_->pool, [this](auto) {
        run_once();
        start_timer();
    })();
}

void plan_queue::run_once()
{
    lock_t lock(mux_);
    auto active_count = pending_tasks_.size();
    for (size_t i = 0; i < active_count; ++i)
    {
        auto rpop_id_next = pending_tasks_.front();

        auto&& it = map_.find(rpop_id_next);
        if (it == map_.end())
        {
            pending_tasks_.pop_front();
            continue;
        }

        auto start_res = handler_(it->second);

        // XXX drop already running?
        switch (start_res)
        {
        case start_result_err_all_limit:
            return;
        case start_result_ok:
            pending_tasks_.pop_front();
            break;
        default:
            pending_tasks_.pop_front();
            pending_tasks_.push_back(rpop_id_next);
        }
    }
}

} // namespace yrpopper::scheduler
