#pragma once

#include <set>
#include <map>
#include <common/config.h>
#include <yplatform/module.h>
#include <boost/date_time/posix_time/posix_time.hpp>

namespace yrpopper { namespace scheduler {

struct TimePointsCounter
{
    void touch()
    {
        auto now = boost::posix_time::second_clock::local_time();
        auto lastMinute = now - boost::posix_time::minutes(1);
        std_lock lock(mapMutex);
        cleanOld();
        if (minutesMap.empty() || minutesMap.rbegin()->first < lastMinute)
        {
            minutesMap[now] = 1;
        }
        else
        {
            minutesMap.rbegin()->second++;
        }
    }

    size_t count()
    {
        size_t result = 0;
        std_lock lock(mapMutex);
        cleanOld();
        for (const auto& item : minutesMap)
        {
            result += item.second;
        }
        return result;
    }

protected:
    void cleanOld()
    {
        if (minutesMap.empty()) return;
        auto timeFrom = boost::posix_time::second_clock::local_time() - timeout;
        while (!minutesMap.empty() && minutesMap.begin()->first < timeFrom)
        {
            minutesMap.erase(minutesMap.begin());
        }
    }

    std::mutex mapMutex;
    std::map<boost::posix_time::ptime, size_t> minutesMap;
    const boost::posix_time::time_duration timeout = boost::posix_time::minutes(15);
};

class module_stat : public yplatform::module_stats
{
    struct chunk_data
    {
        chunk_data() : id(0), count(0)
        {
        }

        chunk_data(const string& m, uint64_t i, uint64_t c) : mdb(m), id(i), count(c)
        {
        }

        string mdb;
        uint64_t id;
        mutable uint64_t count;

        string name() const
        {
            return mdb + ":" + boost::lexical_cast<string>(id);
        }

        bool operator==(const chunk_data& d) const
        {
            return name() == d.name();
        }
        bool operator!=(const chunk_data& d) const
        {
            return name() != d.name();
        }
        bool operator<(const chunk_data& d) const
        {
            return name() < d.name();
        }
        bool operator>(const chunk_data& d) const
        {
            return name() > d.name();
        }
        bool operator<=(const chunk_data& d) const
        {
            return name() <= d.name();
        }
        bool operator>=(const chunk_data& d) const
        {
            return name() >= d.name();
        }
    };

public:
    void add_chunk(const string& mdb, uint64_t chunk_id, uint64_t task_count)
    {
        lock_t lock(mux());
        chunk_data d(mdb, chunk_id, task_count);
        std::set<chunk_data>::const_iterator i = chunks_.find(d);
        if (i == chunks_.end()) chunks_.insert(d);
        else
            i->count = task_count;
    }

    void release_chunk(const string& mdb, uint64_t chunk_id)
    {
        lock_t lock(mux());
        chunks_.erase(chunk_data(mdb, chunk_id, 0));
    }

    void on_task_finished(popid_t)
    {
        finishedTasksCounter.touch();
    }

protected:
    virtual ptree_ptr core_get_stat() const
    {
        ptree_ptr result = yplatform::module_stats::core_get_stat();
        std::stringstream stream;
        for (std::set<chunk_data>::iterator i = chunks_.begin(), i_end = chunks_.end(); i != i_end;
             ++i)
        {
            if (i != chunks_.begin()) stream << ", ";
            stream << i->mdb << ":" << i->id;
        }
        result->put("chunks", stream.str());
        result->put("tasks15", boost::lexical_cast<std::string>(finishedTasksCounter.count()));
        return result;
    }

    std::set<chunk_data> chunks_;
    mutable TimePointsCounter finishedTasksCounter;
};

}}
