#pragma once

#include <common/config.h>
#include <common/context.h>
#include <boost/assign/list_of.hpp>

namespace yrpopper { namespace scheduler {

enum start_result
{
    start_result_ok,
    start_result_err_penalty,
    start_result_err_first,
    start_result_err_host_limit,
    start_result_err_all_limit,
    start_result_err_already
};

class task_index
{
public:
    typedef boost::mutex mutex_t;
    typedef boost::unique_lock<mutex_t> lock_t;

private:
    struct impl
    {
        impl(task_ptr t) : task(t), released(false), paused(false)
        {
        }

        task_ptr task;
        task_ptr update;
        rpop_context_ptr ctx;
        bool released;
        bool finished;
        bool paused;
        mutable mutex_t mux;
    };

public:
    task_index(task_ptr t) : impl_(new impl(t))
    {
    }

    bool start(const rpop_context_ptr& ctx)
    {
        if (impl_->released || impl_->ctx || impl_->paused) return false;
        impl_->ctx = ctx;
        return true;
    }

    void finish()
    {
        lock_t lock(impl_->mux);
        if (impl_->update)
        {
            impl_->task->refresh_data(impl_->update);
            impl_->update.reset();
        }
        impl_->ctx.reset();
    }

    void pause()
    {
        lock_t lock(impl_->mux);
        if (impl_->ctx) impl_->ctx->cancel();
        impl_->paused = true;
    }

    void release()
    {
        lock_t lock(impl_->mux);
        if (impl_->ctx) impl_->ctx->cancel();
        impl_->released = true;
    }

    bool is_released() const
    {
        lock_t lock(impl_->mux);
        return !impl_->ctx;
    }

    rpop_context_ptr ctx() const
    {
        lock_t lock(impl_->mux);
        return impl_->ctx;
    }

    task_ptr task() const
    {
        return impl_->task;
    }

    void refresh(const task_ptr& t)
    {
        lock_t lock(impl_->mux);
        impl_->paused = false;
        if (impl_->ctx) impl_->update = t;
        else
            impl_->task->refresh_data(t);
    }

    void refresh(const task_status_ptr& status, uint64_t is_on)
    {
        lock_t lock(impl_->mux);
        impl_->task->refresh_data(status, is_on);
    }

    mutex_t& mux()
    {
        return impl_->mux;
    }

private:
    boost::shared_ptr<impl> impl_;
};

}}
