#include "module.h"

#include <ymod_smtpclient/call.h>
#include <yplatform/find.h>

namespace yrpopper::smtp {

class module::impl : public yplatform::log::contains_logger
{
public:
    impl(const yplatform::ptree& conf)
    {
        smtpHost = conf.get<std::string>("smtp_host");
        smtpPort = conf.get<uint16_t>("smtp_port");
    }

    Future<SMTPResult> sendMessage(
        rpop_context_ptr ctx,
        const std::string& email,
        const yplatform::zerocopy::segment& message)
    {
        using namespace ymod_smtpclient;

        Promise<SMTPResult> res;

        auto smtpClient = yplatform::find<Call>("smtp_client");
        auto smtp_request = Request();
        auto& address = smtp_request.address;
        address.proto = SmtpPoint::Proto::smtp;
        address.host = smtpHost;
        address.port = smtpPort;

        smtp_request.mailfrom = MailFrom(email);
        smtp_request.rcpts.push_back(RcptTo(email));
        smtp_request.message = std::make_shared<std::string>(message.begin(), message.end());

        smtpClient->asyncRun(
            ctx, smtp_request, [res](ymod_smtpclient::error::Code err, Response response) mutable {
                std::string smtpResponse = response.session ? response.session->data : "";
                if (err)
                {
                    res.set_exception(
                        SMTPException(ymod_smtpclient::error::message(err), smtpResponse));
                }
                else
                {
                    SMTPResult smtpRes;
                    smtpRes.response = smtpResponse;
                    res.set(std::move(smtpRes));
                }
            });

        return res;
    }

    std::string smtpHost;
    uint16_t smtpPort;
};

module::module(const yplatform::ptree& conf) : impl_(std::make_shared<module::impl>(conf))
{
}

Future<SMTPResult> module::sendMessage(
    rpop_context_ptr ctx,
    const std::string& email,
    const yplatform::zerocopy::segment& message)
{
    return impl_->sendMessage(ctx, email, message);
}

void module::logger(const yplatform::log::source& logger)
{
    contains_logger::logger(logger);
    impl_->logger(logger);
}

}

#include <yplatform/module_registration.h>
DEFINE_SERVICE_OBJECT(yrpopper::smtp::module);
