#pragma once

#include <common/types.h>

namespace collectors {

enum class collector_state : int
{
    created = 0,
    ready,
    disabled,
    deleted,
    migrated,
    unmigrated,

    COUNT
};

inline const std::string& to_string(collector_state state)
{
    static constexpr std::initializer_list<const char*> names = {
        "created", "ready", "disabled", "deleted", "migrated", "unmigrated",
    };

    static_assert(
        names.size() == static_cast<size_t>(collector_state::COUNT),
        "collector states count doesn't correspond with collector states names");
    static const std::vector<std::string> state_names(names.begin(), names.end());
    return state_names[static_cast<int>(state)];
}

struct collector_info
{
    collector_info() : id()
    {
    }

    collector_info(const collector_id& collector_id, const uid& dst_uid, const uid& src_uid)
        : id(collector_id), dst_uid(dst_uid), src_uid(src_uid)
    {
    }

    collector_id id;
    uid dst_uid;
    uid src_uid;

    std::string auth_token;

    std::time_t creation_ts = 0;
    fid root_folder_id = EMPTY_ID;
    lid label_id = EMPTY_ID;
    bool ignore_folders_struct = false;

    mid last_mid = EMPTY_ID;
    mids skipped_mids;
    uint64_t old_popid = 0;
    std::string original_server;

    collector_state state = collector_state::created;
    collector_state migration_target_state = collector_state::created;
    std::time_t last_run_ts = std::time(nullptr);
};

using collector_info_ptr = std::shared_ptr<collector_info>;
using collector_info_cb = std::function<void(error, const collector_info&)>;
using collector_info_chunk = std::vector<collector_info>;
using collector_info_chunk_cb = std::function<void(error, const collector_info_chunk&)>;

struct new_collector_draft
{
    uid dst_uid;
    uid src_uid;
    std::string auth_token;
    fid root_folder_id;
    lid label_id;
};

struct migrated_collector_draft : new_collector_draft
{
    bool ignore_folders_struct;
    mid last_mid;
    mids skipped_mids;
    uint64_t old_popid;
    time_t creation_ts;
    std::string original_server;
    collector_state target_state;
};

}
