#pragma once

#include <yplatform/error.h>

#include <string>
#include <vector>

namespace collectors {

using error = yplatform::error;

enum class code : int
{
    ok = 0,
    macs_error,
    sharpei_error,
    operation_exception,
    cancelled_context, // XXX needed? operation_aborted?
    mailbox_access_error,
    validator_error,
    invalid_auth_token,
    no_auth_token,
    get_auth_token_error,
    invalid_grant,
    invalid_login_or_password,
    expired_password,
    password_change_required,
    deleted_collector,
    streamer_data_not_found, // XXX streamer_not_found / no_streamer?
    user_released,
    not_supported,
    passport_error,
    rpop_error,
    node_not_found,
    not_allowed_tvm_source,
    tvm_error,
    empty_email,
    user_not_found,
    download_message_error,
    store_message_error,
    message_is_duplicate,
    duplicate_collector,
    collector_from_himself,
    no_more_messages,
    wrong_login,
    wrong_social_task,

    COUNT
};

inline const std::string& error_message(code c)
{
    static constexpr std::initializer_list<const char*> names = { "ok",
                                                                  "macs_error",
                                                                  "sharpei_error",
                                                                  "operation_exception",
                                                                  "cancelled_context",
                                                                  "mailbox_access_error",
                                                                  "validator_error",
                                                                  "invalid_auth_token",
                                                                  "no_auth_token",
                                                                  "get_auth_token_error",
                                                                  "invalid_grant",
                                                                  "invalid_login_or_password",
                                                                  "expired_password",
                                                                  "password_change_required",
                                                                  "deleted_collector",
                                                                  "streamer_data_not_found",
                                                                  "user_released",
                                                                  "not_supported",
                                                                  "passport_error",
                                                                  "rpop_error",
                                                                  "node_not_found",
                                                                  "not_allowed_tvm_source",
                                                                  "tvm_error",
                                                                  "empty_email",
                                                                  "user_not_found",
                                                                  "download_message_error",
                                                                  "store_message_error",
                                                                  "message_is_duplicate",
                                                                  "duplicate_collector",
                                                                  "collector_from_himself",
                                                                  "no_more_messages",
                                                                  "wrong_login",
                                                                  "wrong_social_task" };

    static_assert(
        names.size() == static_cast<size_t>(code::COUNT),
        "Error codes count doesn't correspond with error names count");
    static const std::vector<std::string> error_names(names.begin(), names.end());

    return error_names[static_cast<int>(c)];
}

enum class errc : int
{
    ok = 0,
    task_expired,

    COUNT
};

inline const std::string& errc_message(errc err)
{
    static constexpr std::initializer_list<const char*> names = { "success", "task_expired" };
    static_assert(
        names.size() == static_cast<size_t>(errc::COUNT),
        "Error conditions count doesn't correspond with condition names count");
    static const std::vector<std::string> condition_names(names.begin(), names.end());

    return condition_names[static_cast<int>(err)];
}

}

namespace yplatform {

template <>
struct is_error_enum<collectors::code>
{
    static const bool value = true;
};

}

namespace boost { namespace system {

template <>
struct is_error_condition_enum<collectors::errc>
{
    static const bool value = true;
};

}}

namespace collectors {

const boost::system::error_category& get_error_category();
const boost::system::error_category& get_error_condition_category();

inline const boost::system::error_category& error_category(code /*e*/)
{
    return get_error_category();
}

inline boost::system::error_condition make_error_condition(errc e)
{
    return boost::system::error_condition(static_cast<int>(e), get_error_condition_category());
}

inline std::string error_message(error ec)
{
    return ec ? ec.message() : "ok";
}

inline error make_error(std::exception_ptr ptr) noexcept
{
    if (!ptr) return error(code::operation_exception, "Bad exception");
    try
    {
        std::rethrow_exception(ptr);
    }
    catch (const std::exception& e)
    {
        return error(code::operation_exception, e.what());
    }
    catch (...)
    {
        return error(code::operation_exception, "Unknown exception");
    }
}

}
