#pragma once

#include <common/collector_info.h>
#include <common/message.h>
#include <common/types.h>
#include <common/user_info.h>

#include <yplatform/log.h>
#include <yplatform/module.h>
#include <yplatform/find.h>

namespace collectors::typed {

namespace tl = yplatform::log::typed;

inline tl::attributes_map make_common_attrs(
    context_ptr ctx,
    const std::string& type,
    const global_collector_id& collector)
{
    tl::attributes_map am;
    am << tl::make_attr("y_context", ctx->uniq_id()) << tl::make_attr("type", type)
       << tl::make_attr("collector", collector.to_string())
       << tl::make_attr("dst_uid", collector.uid);
    return am;
}

inline tl::attributes_map make_status_attrs(const error& ec)
{
    tl::attributes_map am;
    am << tl::make_attr("status", ec ? "error" : "success")
       << tl::make_attr("reason", error_message(ec));
    return am;
}

inline void log_store_message(
    context_ptr ctx,
    const global_collector_id& collector,
    const user_info& src_user,
    const user_info& dst_user,
    const message& src_message,
    const message& dst_message,
    const error& ec)
{
    yplatform::log::tskv_logger log(YGLOBAL_LOG_SERVICE, "typed_log");
    YLOG(log, info) << make_common_attrs(ctx, "store_message", collector) << make_status_attrs(ec)
                    << tl::make_attr("dst_email", dst_user.email)
                    << tl::make_attr("dst_fid", dst_message.fid)
                    << tl::make_attr("dst_mid", dst_message.mid)
                    << tl::make_attr("src_uid", src_user.uid)
                    << tl::make_attr("src_email", src_user.email)
                    << tl::make_attr("src_fid", src_message.fid)
                    << tl::make_attr("src_message_id", "mid:" + src_message.mid)
                    << tl::make_attr("src_type", "yandex")
                    << tl::make_attr("hdr_message_id", src_message.hdr_message_id);
}

inline void log_api(
    context_ptr ctx,
    const global_collector_id& collector,
    const std::string& method,
    const error& ec,
    const std::string& info = "")
{
    yplatform::log::tskv_logger log(YGLOBAL_LOG_SERVICE, "typed_log");
    YLOG(log, info) << make_common_attrs(ctx, "api", collector) << make_status_attrs(ec)
                    << tl::make_attr("method", method) << tl::make_attr("info", info);
}

inline void log_iteration(
    context_ptr ctx,
    const collector_logging_info& collector_info,
    collector_state state,
    std::time_t start_ts,
    std::size_t sent_count,
    const mid& last_mid,
    const error& ec)
{
    yplatform::log::tskv_logger log(YGLOBAL_LOG_SERVICE, "typed_log");
    YLOG(log, info) << make_common_attrs(ctx, "iteration", collector_info.id)
                    << make_status_attrs(ec) << tl::make_attr("src_uid", collector_info.src_uid)
                    << tl::make_attr("duration", std::time(0) - start_ts)
                    << tl::make_attr("validated", state == collector_state::ready)
                    << tl::make_attr("src_type", "yandex")
                    << tl::make_attr("sent_count", sent_count)
                    << tl::make_attr("last_mid", last_mid);
}

}
